rng = Random.default_rng()
Random.seed!(99)

#No Noise forecasting from eta = 0 to the 90th element of the domain etaspan = (0.05, 5.325)


#Constants
C = 0.01


#Initial Conditions
I = [1.0, 0.0]   #Psi(0)=1, Psi'(0)=1
etaspan = (0.05, 5.325)

#radius range
datasize= 100
etasteps = range(etaspan[1], etaspan[2]; length = datasize)


function whitedwarf(du, u, p, r)
    psi = u[1]
    dpsi = u[2]
    du[1] = dpsi
    du[2] = (-((psi^2-C))^(3/2) - 2/r * dpsi)
end


#Defining the Ordinary differential equation as an ODEProblem with the DifferentialEquations.jl
prob = ODEProblem(whitedwarf, I, etaspan)
#Solving the ODEProblem with the Tsit5() algorithm
sol = solve(prob,saveat=etasteps)


dudt2 = Lux.Chain(Lux.Dense(2, 80, tanh),Lux.Dense(80, 80, tanh), Lux.Dense(80, 2))
#Setting up the NN parameters randomly using the rng instance
p, st = Lux.setup(rng, dudt2)

etasteps =  etasteps[1:end-60]
etaspan = (etasteps[1], etasteps[end])
I=[1.0,0.0]
prob_neuralode = NeuralODE(dudt2, etaspan, Tsit5(); saveat = etasteps)

function predict_neuralode(p)
    Array(prob_neuralode(I, p, st)[1])
end
#Training data
true_data= Array(sol[:,1:end-60])
### Define loss function as the difference between actual ground truth data and Neural ODE prediction
function loss_neuralode(p)
    pred = predict_neuralode(p)
    loss = sum(abs2, true_data .- pred)
    return loss, pred
end


p_trained_nonoise = (layer_1 = (weight = Float32[0.23796381 0.18667705; -0.28679243 -0.26371777; -0.35759655 -0.12012835; 0.088092715 -0.16617161; 0.2127927 0.23096359; -0.22491702 -0.35370404; -0.1431448 -0.0319932; -0.12282468 -0.02371963; -0.42553076 -0.29566896; -0.18892774 0.1146335; -0.31919464 0.16279352; 0.07911945 -0.06653185; 0.23981288 0.08612904; 0.07598175 -0.34019312; 0.119357064 -0.1359894; 0.49998587 0.106471226; -0.14746168 0.062903665; 0.04550562 -0.45156586; -0.053310856 0.009037496; 0.021229157 0.15313466; -0.19980381 0.1613393; 0.18346138 -0.08696011; -0.21861263 0.08263907; -0.16086522 0.11793756; 0.3843232 0.3623592; -0.24072061 -0.028373009; 0.0858469 -0.13332525; -0.15346608 0.20589206; 0.24343221 0.012604643; 0.38382694 0.31354037; -0.017082192 -0.1467851; -0.066396214 -0.3270168; -0.09737127 -0.048304494; -0.2865124 0.16079265; 0.0135186715 -0.41716707; -0.1393893 -0.19121593; 0.034893773 -0.04363638; -0.16614433 0.3287394; 0.23171858 -0.1461355; 0.10227923 -0.28960696; -0.2026282 0.19716005; 0.031516295 -0.2242235; -0.26406884 -0.24501689; 0.038934667 -0.21643361; -0.116312064 0.18743622; -0.04289466 -0.12533472; -0.2586438 0.17625237; -0.053815834 -0.029424686; -0.25473574 -0.18674238; 0.18385352 0.018658811; -0.30878302 -0.15481679; -0.049035594 0.02429007; 0.0089704525 -0.19175361; 0.020343643 0.291479; -0.27536833 -0.11665913; 0.23274572 0.01773306; 0.24403478 0.14547947; 0.07443597 -0.07884615; -0.004789191 -0.29739285; -0.008610432 0.5437783; -0.103571735 0.370282; 0.23785481 0.033570725; 0.15195343 -0.11592756; -0.11396957 0.05850431; 0.04757107 -0.43371278; 0.2650788 0.12317446; 0.15704444 -0.090546824; 0.2249284 0.10919313; -0.23914036 -0.011396494; 0.31788796 0.16040714; 0.29940975 0.2754855; 0.2690076 -0.04870999; -0.25906998 -0.23891404; -0.39230305 -0.24726695; 0.04577174 -0.29699475; -0.051317014 0.18871997; 0.18442951 -0.07142783; -0.113083795 0.09079125; 0.1701407 -0.09663324; 0.13501063 -0.19747195], bias = Float32[0.045973957; 0.037968606; 0.044585932; -0.1397955; -0.0033801484; -0.011736001; 0.06399012; -0.028850395; 0.07927133; 0.08305366; 0.0496512; -0.1032204; -0.010915226; -0.099443845; -0.1066075; 0.056876395; 0.10785918; -0.04832909; 0.07029531; 0.1063861; 0.080745496; -0.15193719; 0.017532269; 0.08278143; 0.051971897; 0.12584975; -0.081115365; 0.08921613; 0.01077011; -0.03904921; 0.01904736; -0.12128136; 0.022054752; 0.09806216; -0.111316256; 0.010422081; -0.12737814; 0.043228578; 0.016165731; -0.109338224; 0.054840844; -0.042817637; 0.031583067; -0.118142284; 0.14122632; 0.070675045; 0.039836247; -0.064323485; 0.06872568; 0.00080986996; 0.09839707; 0.025358481; -0.064807676; 0.12433512; 0.031565968; -0.008531598; -0.030216448; -0.079520024; -0.10093742; 0.037164062; 0.120194; -0.08895879; -0.009927845; 0.08854883; -0.09519153; 0.008556435; -0.115752384; -0.013252497; 0.104311846; -0.03957599; -0.09950894; -0.060354896; 0.03241506; -0.04067396; -0.061569393; 0.11136245; 0.015641483; 0.12773901; -0.0696075; -0.071187206]), layer_2 = (weight = Float32[-0.12902567 0.21739328 0.11899069 -0.21017636 -0.19858427 0.34897014 -0.011318961 -0.16289884 0.2347338 0.1743011 0.033735443 0.09159604 -0.20962654 0.0397803 0.118548356 -0.014144386 -0.008769436 0.104779825 0.11239722 0.15323578 0.19739641 -0.15792385 -0.015977798 0.23592316 -0.35813957 0.10377673 -0.20629515 0.025996847 -0.09614222 -0.2205986 0.2691166 0.12938713 0.1448679 0.090249 -0.10068839 0.19517985 -0.17160732 -0.008983539 -0.08254377 -0.20228083 -0.05156407 -0.033082627 0.01538536 0.13580292 -0.060029477 0.011631554 0.043864593 -0.20214461 0.21786271 -0.20571113 -0.044859055 0.23579334 -0.118826516 -0.059233606 0.23251419 -0.07765515 0.025473002 -0.1794975 0.05152868 -0.17330478 -0.07617303 -0.21685661 -0.17500393 0.07460968 -0.07134558 -0.22527508 -0.066303425 -0.19877976 0.20622566 0.015444454 -0.21551843 -0.24758561 0.34650505 0.009979777 -0.055834644 -0.0016615374 -0.057442877 0.096464396 -0.12502345 -0.028509976; -0.0065611918 0.02234447 0.050320007 -0.16230763 -0.32388836 0.32337055 0.037496656 -0.08381812 -0.056120407 0.18556298 -0.006216117 0.056295566 -0.06255598 0.0081607 0.10675289 -0.013078906 0.28585622 -0.039074402 0.14342082 0.06360035 0.17899735 0.0153210405 0.06353223 0.069957376 -0.31640422 0.07426363 -0.111724675 0.042157095 0.035772834 -0.17235334 0.13983026 0.046923026 0.14378284 0.08730726 -0.08959766 0.030959869 0.0031393308 0.045676503 -0.13507643 -0.20494007 0.13735674 0.06660596 -0.034544062 0.053610638 0.059905794 -0.022962404 0.14639579 -0.04669531 0.10950059 -0.1906711 -0.06306703 0.08207198 0.015070041 0.11280373 0.059133276 -0.03911045 -0.2663444 -0.26688224 -0.12707892 0.11509702 -0.01550154 -0.20815165 -0.17479506 0.2185821 -0.19166586 -0.2022003 0.034860063 -0.02109554 0.27968556 -0.3193592 -0.23008285 -0.07621563 0.18614547 0.04236544 -0.056347452 0.06701916 0.1353855 0.18508708 -0.26345026 -0.24472305; 0.1530365 0.07593288 -0.11883973 0.16463755 0.006430016 0.092257306 0.12611465 -0.119132355 -0.11773299 -0.08223479 -0.07486492 0.11102536 0.09003456 0.019625414 -0.065937296 -0.12191907 -0.2881104 -0.018231547 0.026812356 -0.133361 -0.30128852 0.0945615 -0.07873301 -0.16680355 -0.18800966 0.00093591056 0.08070439 -0.18152514 -0.01663241 0.06942455 -0.10511632 0.015290267 -0.10983847 -0.27959472 -0.01823799 -0.22894633 0.086867675 -0.4026076 0.15203823 0.1820552 0.03152689 0.26983428 0.13098766 0.07282505 -0.14763115 -0.12515435 -0.2522971 0.048057362 -0.16157107 -0.012654601 -0.19001465 -0.014859428 0.12071601 -0.3268146 -0.33435658 0.11697953 0.18196177 0.20448388 0.1943155 -0.20208383 -0.033875305 0.3456303 0.21606149 -0.13112554 0.08689277 0.09332751 0.14074522 0.167221 -0.23772044 -0.12548961 0.04798187 -0.064548135 0.1956255 -0.035354234 0.082678095 -0.05693398 0.23886228 -0.1352369 0.24582227 0.41435608; 0.114400275 -0.1712971 0.069720015 0.2658091 0.09176202 -0.035420462 -0.3335553 -0.114505365 -0.29367125 -0.15069795 -0.24729551 -0.04216085 0.22829299 0.12848346 -0.07138958 -0.15524077 -0.04788955 0.14473367 -0.1563045 -0.050995488 -0.18475789 0.20555845 -0.22425535 -0.23211706 -0.17090453 -0.2503681 0.27532354 0.033623077 0.16460855 0.24445751 -0.006695506 0.06659816 0.006501575 -0.2876279 0.02497272 -0.24466604 0.29027644 -0.116283245 0.043458063 0.055333655 -0.17665507 0.21644814 -0.20759895 0.30169368 -0.1487285 0.046429433 0.061269466 0.2663603 -0.023191333 0.028162323 0.018685203 -0.1782769 0.16248108 -0.3370267 -0.05429209 0.23541154 0.19233477 0.10370598 -0.023782937 -0.15846208 -0.2836133 0.30256516 0.18010782 -0.017230127 0.16847016 0.13336025 0.07256014 0.015167924 -0.012584299 -0.12613852 0.10772658 0.17936568 -0.21164507 -0.17652957 0.14810352 -0.21296968 0.16167021 -0.23253952 0.18143578 0.08932409; 0.059934642 0.24906956 0.053696223 -0.31711665 -0.19783586 -0.04139177 0.0229345 0.13449222 0.0785113 0.18724392 0.1858303 -0.23997368 0.05122128 -0.18613175 -0.19914474 -0.11894508 0.14847106 -0.06616674 -0.032691386 0.024126772 0.31387663 -0.042355273 0.17226554 0.3067135 0.13802682 0.26126075 -0.07364206 0.06428116 0.030698113 -0.112711355 -0.085921116 -0.24950874 0.2246983 0.18511374 -0.34650278 0.18212715 -0.21017273 0.22682402 -0.325731 -0.14688043 0.14184286 -0.13381514 -0.06931323 -0.23356518 0.22348025 -0.02450384 0.27051333 0.038270902 0.149844 -0.1708083 0.2756452 -0.015756851 -0.014911163 0.2909774 0.10308642 0.043445658 -0.047464166 -0.009612367 -0.12887305 0.142733 0.057051796 -0.00080855 -0.13273126 0.20040737 -0.20813128 -0.11577371 0.01273367 -0.30377474 0.20339029 -0.19234474 -0.017011002 -0.10716704 0.17880243 0.26495728 -0.12700848 0.2061206 -0.16621761 0.11666935 -0.15626763 0.005115339; 0.3193215 -0.24522917 -0.27451226 0.18271746 0.035854015 -0.0064881546 0.032905933 0.10843183 0.029595193 -0.038310632 0.034916706 0.044880547 0.063956134 -0.05076357 0.026666328 0.21521354 -0.03642013 0.016869832 0.024693497 -0.3363311 0.15706599 0.0047018975 -0.18450825 -0.032875996 0.14813475 -0.31533048 -0.060279027 -0.30563223 0.2552899 0.32935807 -0.22052571 0.024070006 -0.04882363 0.06442201 -0.15733373 -0.0067131845 -0.08775861 -0.28126362 -0.050710488 0.20629214 -0.0059423004 0.12532093 -0.25812817 -0.049833246 -0.006995728 -0.32057017 -0.01632654 0.08667934 -0.06642785 0.22357589 -0.26425472 -0.29044497 0.090304665 0.059972618 -0.090942 0.25257093 -0.07433052 0.12695064 -0.23312506 0.11679948 -0.19759789 0.034418616 -0.037995968 -0.100598976 -0.0046880958 0.15507509 0.2512303 0.2800588 -0.14208983 0.034756385 0.021530615 -0.09253854 -0.043251067 -0.24624841 0.055555854 -0.020354912 0.09418949 -0.2284668 0.040246565 0.14351498; 0.13266179 -0.30271292 -0.037337285 -0.16886014 -0.020843726 -0.30351046 -0.09570621 0.10754938 -0.066984 0.076581106 0.14210571 -0.19144389 -0.07142841 -0.01237591 -0.080538586 -0.0028292476 -0.17873342 -0.09902237 -0.067731045 -0.06701334 0.12379865 0.17147978 0.05345714 0.032424405 0.23174559 0.056831505 -0.14821789 0.05276559 0.08237526 0.089420624 -0.22207615 -0.09574318 -0.20602262 -0.059002366 -0.21049747 -0.16775407 0.20737115 -0.02546666 -0.13144794 0.15083335 0.24683565 -0.059698973 -0.104869016 0.08548317 0.010481088 -0.19700778 -0.08114426 -0.022853341 -0.03436876 -0.03827881 -0.17497087 0.055012558 0.21859702 -0.043629687 -0.13921726 -0.1421597 0.09393182 0.18560843 -0.11934883 0.11901606 -0.05322147 0.27120996 -0.13294116 -0.21286969 -0.22523537 0.029686153 -0.07536065 0.1880261 0.00073954975 0.003059079 0.0077590686 0.13404489 -0.16609727 -0.32916483 0.0571059 0.13709436 -0.21685284 -0.10983657 0.12603585 -0.06931398; -0.14192577 -0.0067446316 0.008495166 0.06124226 -0.14908451 0.12757938 -0.031940512 -0.047142353 0.016024059 -0.12193521 -0.21473065 -0.067414954 -0.10250883 0.15744814 0.008654252 -0.026061496 -0.17834908 0.2524657 0.016917726 0.26847792 0.09727351 -0.01450209 0.055742465 0.14258732 -0.30119574 0.096983165 0.118775 -0.044807978 -0.13637172 -0.21670276 -0.040418096 0.22568569 0.17612696 -0.19786334 0.08747943 0.24870713 0.17144223 -0.18262105 0.088528216 0.037183955 0.11070665 -0.058375996 0.22059144 0.25552726 -0.11895052 0.29509613 -0.13930184 -0.12327985 0.275165 -0.121358536 0.023052527 -0.11934956 0.07840335 0.08551954 0.15972763 -0.0021458683 0.103294164 -0.057678185 0.024214182 -0.09592268 -0.26985025 -0.14193378 -0.055575542 0.052761696 0.16028465 -0.21428216 -0.14401293 -0.063270606 0.14626639 -0.104969524 -0.25047594 0.19358031 0.04795641 0.21855393 0.18035525 -0.1359715 0.11592482 0.0011369144 0.08440812 -0.0130410995; -0.21444486 0.140391 0.09394859 -0.01002465 -0.26900178 0.31335264 0.12636435 -0.095041126 0.04917392 0.12519878 0.13168527 0.05848064 -0.17537616 -0.21744351 -0.11746786 -0.02739534 0.15789886 -0.061812498 0.28733674 0.07026804 -0.08856494 -0.070266806 0.11966319 0.124383934 -0.24965566 0.17171356 0.058105446 0.30082896 0.03178579 -0.03646804 -0.014590572 -0.19400904 -0.060936328 -0.05248153 0.09514414 0.043376744 -0.0030524968 0.22557648 0.11661557 0.0041658273 -0.010435874 0.07407284 0.16173181 -0.04562467 -0.023650445 0.04150508 0.26663703 -0.11026399 0.20041725 -0.1278225 0.22102198 0.23495197 -0.072893694 -0.058241647 0.07600291 -0.24884552 -0.2702409 -0.13128865 0.09114851 -0.046904135 -0.032720175 -0.13036563 -0.08154548 0.22854939 -0.07594162 -0.019774761 0.03106285 0.069770545 0.16102308 -0.22237581 -0.1264434 0.08548375 0.07321637 0.04234244 -0.240767 0.19484635 0.08815063 -0.038538437 -0.17422514 -0.10850847; -0.05274394 0.16050568 0.27849004 -0.16679099 -0.12385136 0.26037005 0.069169275 -0.1314402 0.27295148 0.17704423 0.18934938 -0.017298788 -0.07336706 -0.17881496 -0.24967304 -0.00085582194 -0.017712336 -0.15681702 0.2890775 0.08900314 -0.06914279 -0.22086525 0.077646434 0.022245534 -0.0008352556 -0.021231772 0.04812571 0.14915869 -0.20169929 -0.32290182 0.26588365 -0.31971127 -0.010007463 -0.0075811143 0.031712808 0.12563853 -0.07818162 0.12916635 -0.07584747 -0.23283602 0.25718996 -0.13023475 0.19493249 -0.19649786 0.09097368 0.20602737 0.0089890575 -0.121854104 0.1258946 -0.23079845 -0.044120193 0.29681844 -0.23332436 0.14813758 0.30266517 -0.21221803 -0.15077943 -0.16085145 0.0962939 -0.019069491 0.28324583 -0.24506795 -0.2542996 0.15258728 -0.088283435 -0.09225537 -0.13372777 -0.09322149 0.15935428 0.054545455 -0.21023719 -0.29199135 0.2034625 0.0489506 -0.15466206 0.05158723 0.14331083 -0.045315593 -0.23021951 -0.2527316; -0.02357774 -0.065696485 -0.05133728 0.27081352 -0.12493791 0.06765177 -0.17862579 -0.20593055 0.049506743 -0.34205604 -0.1993547 0.23333468 0.2633579 0.16829404 0.08942786 0.06394411 -0.053252492 0.3156846 -0.18133111 -0.014071566 -0.15406528 0.29093954 -0.18768084 -0.07130525 -0.12684913 -0.26611575 0.04719171 -0.32684547 0.122707106 -0.027600463 0.17611952 0.29717717 -0.026557382 -0.060418263 0.16976133 -0.022178322 0.1093709 -0.18252464 0.06010907 0.3208413 -0.19246402 0.07264183 0.13431787 0.22543713 -0.30587667 0.09584233 -0.30523998 -0.1090217 -0.05085301 0.2109639 -0.18453482 -0.12118177 0.17078194 -0.3772045 0.02035013 0.2567024 0.07460567 0.14410886 0.2236765 -0.3080196 -0.3449369 0.254457 0.21302596 0.068745695 0.29969713 -0.10408984 0.18294342 0.1608805 -0.13195649 -0.10773224 0.00434175 0.29983598 0.04412347 -0.22327071 0.08997207 -0.16907048 0.059014536 -0.1303504 0.19160378 0.37851056; -0.016416619 -0.03865514 0.015657386 -0.21827199 -0.1453514 -0.19524714 0.022557313 0.06498739 0.08294216 0.01632379 0.09429442 -0.02909126 -0.19925609 -0.0730782 -0.31392607 -0.023051241 0.29441905 -0.076808445 0.0792394 0.05084373 0.15129247 -0.22725019 0.22224814 0.134856 0.07203747 -0.037390936 -0.08280823 -0.007497798 -0.1787561 -0.052259393 0.118092775 -0.14302708 0.12825865 -0.00088610296 -0.17048375 0.21744023 -0.1877172 0.2212277 -0.31591356 -0.270578 0.2501505 -0.28413147 0.014136296 -0.07675958 0.1360019 0.23056304 0.12605835 0.15020993 0.18213406 -0.18245153 -0.046502504 0.001324901 0.038642444 0.10268231 0.3215652 -0.19656289 -0.2568728 -0.16254166 -0.106178686 0.029208459 0.16487035 -0.16136359 -0.31868792 0.047899 -0.03470198 -0.3023247 -0.26416194 -0.027190994 0.29804415 0.017750729 0.11771993 -0.19235477 0.027784763 -0.058539674 -0.09033132 0.045634557 -0.26722118 0.18873613 -0.092218846 -0.1449694; -0.0446586 0.31022298 0.115242526 -0.085629426 -0.025242923 0.105132155 0.06593647 -0.1296896 0.3115685 -0.03164363 -0.07270991 0.026998678 -0.2795811 -0.014334071 -0.02218768 -0.045583993 -0.00053286616 -0.099112876 0.19550933 0.1709617 0.07816906 -0.016030528 0.18915516 0.24235287 -0.1192542 0.2890665 -0.09288374 0.28126228 -0.03263832 -0.2947966 0.24452049 -0.07092256 0.26865166 0.1318754 -0.12312748 0.08374916 -0.15917316 0.1598966 -0.052855488 -0.16599655 -0.14898783 0.06993622 0.26985735 -0.0011708015 0.24576752 0.2725554 0.16702087 -0.29482126 -0.04679706 -0.13151902 0.17850214 0.024656195 -0.31042972 -0.1333783 0.2153749 -0.24550426 0.022931065 0.012941888 0.05821423 -0.0012501499 0.15267503 -0.16153418 -0.07543349 0.22301331 -0.038756713 -0.011239486 -0.025757195 -0.2825005 0.043251365 0.08647229 -0.1275736 -0.05328595 0.30234733 0.27155766 -0.16501476 -0.019879777 0.17771788 0.12119684 0.02365582 -0.1641882; -0.31975338 0.06755522 -0.027772356 0.17588186 -0.26731667 0.31881773 0.00033317495 -0.01399626 0.15190415 0.20093799 0.03185832 0.19508077 -0.26781362 0.049564805 0.17134704 -0.23395208 -0.049340345 0.18815482 0.00973083 0.01344929 0.0630794 0.11581061 -0.17011276 0.123493016 -0.03946631 0.0046733026 0.062071234 0.20220058 -0.13055907 -0.21517281 0.15435776 0.21534333 0.18962303 0.15881415 0.056756824 0.3189412 -0.0015042705 0.0030704415 -0.1441841 0.024252161 0.06560082 0.09861248 0.121477365 -0.05468301 0.069787316 -0.056608852 0.18108433 -0.090790085 0.23943913 -0.1423536 0.31593126 0.021262154 -0.30478683 -0.026205294 0.22924517 -0.080183655 -0.06729674 0.066076435 -0.05434767 -0.2174453 -0.0118895015 -0.07647286 0.14863689 0.01591995 0.04827849 -0.07767315 -0.08079282 -0.23853572 0.2045236 -0.041695494 -0.03937456 -0.14419468 0.2203561 0.19870616 -0.204718 0.06493777 0.18224418 0.17053723 -0.09377715 -0.27322695; 0.031338446 -0.23188265 -0.2091016 0.098052405 0.04996914 -0.07591789 -0.079611406 0.115587674 -0.12166701 -0.2940447 -0.13404283 0.030694801 0.06867875 0.10752194 0.13212016 0.0655852 -0.26058766 0.16629517 -0.077431485 -0.1280905 -0.023334188 0.24524216 0.052655928 0.029869994 0.17950231 0.036485635 0.07603613 -0.14731675 0.027483158 0.2691418 0.0021260187 0.059802517 -0.111464985 -0.05622953 0.095199846 -0.12248601 0.17669933 -0.07160409 0.0028949215 -0.0906422 -0.010841699 -0.075058036 -0.06380036 -0.04940131 -0.19738328 -0.06884458 -0.15646258 -0.010808823 -0.027681503 0.1506084 -0.21581234 -0.25596312 0.004017706 -0.123255566 -0.02111811 0.10696926 0.22165619 0.2027379 -0.09465077 -0.21596436 -0.12486656 0.21618976 0.15144835 -0.12895428 0.096038096 0.2583729 0.0048750117 0.21326324 -0.0051361793 0.17212315 0.20672467 0.26868322 -0.22416332 -0.15663947 0.22625338 -0.11097131 -0.19192733 -0.3106398 0.22295399 0.13937879; 0.079410635 0.030413754 -0.19779393 0.21072108 0.049768798 -0.11853668 0.03319686 0.16041362 -0.11971619 -0.28095892 -0.1425133 0.0635028 -0.035056397 -0.13322788 0.11635952 0.12813792 -0.25830457 -0.17650817 0.009030239 -0.15450998 -0.019873286 -0.11936265 0.09637484 -0.29752412 0.040113963 0.01956383 0.11204243 -0.2814971 0.051575918 0.21660985 0.013616036 -0.07111493 -0.10553937 0.076559216 -0.07320475 -0.027390309 0.022981096 -0.11412153 0.1974646 0.06448853 -0.13600978 0.08929595 -0.1482871 0.15347175 -0.11879778 -0.055405993 0.020025317 0.28637174 0.009801694 0.047067214 0.029907176 -0.19475049 0.32116857 0.19485438 -0.24621102 0.010924184 0.30032563 0.09675636 0.06591913 -0.039872095 -0.12276596 0.13867009 -0.1684123 -0.04522872 -0.1023039 0.11564913 0.1408231 0.20643134 -0.11645564 0.09304766 0.33069623 0.10994348 -0.24988098 -0.27140573 0.18864298 -0.19953784 -0.21883549 -0.19787472 -0.034369454 -0.052489355; 0.28911844 -0.25281626 -0.20381038 0.13220742 0.30781603 -0.21447705 -0.16877253 -0.118885905 0.018424397 0.057836164 -0.07803932 0.053280324 -0.07343547 0.02901487 -0.065859474 -0.0058794618 0.03606946 -0.104702674 0.026877705 -0.09698552 -0.21010087 0.2125621 -0.10612067 -0.2548635 0.24321434 -0.052471105 0.20348035 -0.20429707 0.26078895 0.07624537 -0.18962252 0.109512754 -0.24914533 -0.26794422 0.21682356 -0.30182818 0.24265572 -0.017467003 0.057807084 0.13401666 -0.00042438242 0.23795821 -0.21111716 0.24893199 0.071025595 -0.12249187 0.008210345 0.217145 -0.30384952 0.04776745 -0.095924914 -0.19293222 0.013486245 -0.0910089 -0.1725316 0.1706512 0.29410478 -0.067361414 -0.095864944 0.12329607 0.071596146 -0.017881129 0.18936671 -0.26522586 -0.07265933 0.14223595 0.24751389 0.04436235 -0.17673594 0.15699908 0.06829884 0.07274481 -0.22927725 -0.048369884 -0.004756161 0.081367105 -0.04751166 -0.17043898 0.22997142 0.0073031993; 0.052370913 -0.1821823 -0.15806359 -0.071711235 0.27725017 -0.20108946 -0.23455676 -0.1003508 0.02028189 -0.082396045 -0.0062061604 -0.14914763 0.087951705 -0.079210974 -0.20025948 0.096419916 0.020212576 -0.12506807 -0.17125137 -0.16816849 -0.15291552 0.114307 -0.009435816 0.030761117 0.1693163 -0.18769625 0.166691 -0.2561163 0.24230479 0.16357094 -0.27937394 0.08154704 -0.28722587 -0.04253507 -0.15111291 -0.10320147 0.18577988 -0.15958832 0.050631512 -0.006279638 -0.0819962 0.042803466 -0.26733217 0.16546896 0.059316956 -0.16863237 -0.095337324 0.046263203 -0.22882873 -0.12714177 -0.12434263 -0.16284223 0.058704518 0.039862793 -0.083604194 0.26034236 -0.023869358 -0.16149794 -0.09934608 -0.12199575 -0.02403759 0.0878202 -0.13864478 -0.05681675 -0.019265745 0.10888884 -0.064953 0.23395985 -0.22005598 0.31795812 0.18749857 -0.04311067 -0.028530877 -0.14219666 0.27029997 -0.05431664 -0.07753161 -0.049499113 0.038405906 -0.008247566; 0.012552165 -0.26932693 -0.090793215 -0.015664095 0.12851146 -0.050327994 -0.11100033 -0.047220796 -0.324442 0.044999275 -0.019459523 -0.17261617 -0.0011458609 0.201055 -0.1782495 0.05252315 -0.12291478 -0.03512999 -0.04235002 -0.094078764 0.011166136 0.24586882 -0.027672825 -0.12597193 0.28347805 -0.12038476 -0.008499472 0.04612913 -0.0098343175 0.090296894 -0.21052961 -0.12835531 -0.266705 0.10911348 0.20252787 -0.14179681 0.27150413 -0.14074852 0.16028662 0.2025968 0.0056779664 0.011230665 -0.19132668 -0.06684213 -0.15599427 3.629552f-5 0.0840459 0.29995093 -0.22319779 0.022986112 -0.23541395 0.034346286 0.11315996 -0.012813399 0.03813672 0.14324038 0.032777984 -0.06428297 -0.1506115 -0.18439308 -0.15411766 -0.008989706 0.010791735 -0.03274523 -0.084268354 -0.06579671 0.23280388 0.16501729 0.04754103 0.26463905 0.17668155 0.2715764 -0.19805877 -0.092711434 -0.0736595 -0.05443152 0.09718841 -0.059174784 -0.002276796 0.109824315; -0.3187318 0.16140647 0.3106889 -0.13867678 -0.15199041 0.15253374 0.27978978 0.018170172 0.28835225 0.060562607 0.26588458 -0.042579427 -0.1631408 -0.16930924 -0.077429175 -0.06893529 0.018060192 -0.18476945 -0.0078957975 0.24357946 -0.094715185 -0.022965996 -0.081440106 0.23367706 -0.24550821 0.008422032 -0.14787523 -0.060978178 -0.07947434 0.006459038 0.08507735 0.1336356 0.20403324 -0.09314423 0.020694206 0.11342976 -0.121822596 0.07976463 -0.20818296 -0.22654168 0.1371689 0.029245451 0.22118208 0.0020943407 0.015109822 0.23199631 0.13769351 -0.080926135 0.20398061 0.099525034 0.1398854 0.13279252 -0.3149882 -0.04784319 0.2722335 -0.058334474 -0.24155077 -0.28236064 -0.03650833 0.079587996 -0.07435699 0.062241565 -0.11672025 0.008829731 0.014217784 -0.25301543 0.069045596 0.04729806 0.15232173 -0.32704848 -0.13299264 -0.20672534 0.32735938 0.12065612 -0.26924536 -0.046958428 0.05555829 -0.020114265 -0.14691691 -0.04219784; 0.31394646 -0.049897883 -0.25475952 0.113928095 0.32204917 -0.11074217 -0.090135485 -0.004164395 -0.2458536 0.08377669 -0.19345976 -0.043171916 0.099258766 -0.04980908 -0.24952623 0.23519485 -0.1183428 -0.091919705 -0.13920355 -0.18887144 -0.009862135 0.22215903 -0.14977431 -0.14938407 0.08246983 -0.0729937 -0.100233436 -0.30209196 0.08295464 0.012797538 -0.03902632 -0.10325285 0.060279664 -0.13980243 -0.14992096 -0.13522838 -0.10219055 -0.26490903 0.027044173 0.16047509 0.109771684 -0.04695161 -0.03818197 -0.04721451 -0.0005610515 -0.21087752 -0.13586815 0.117414854 -0.04033664 0.16440083 -0.120671086 -0.26954842 0.015780464 0.1472743 -0.035878688 0.020812953 0.1751412 0.036526445 0.10872238 0.010345928 0.1014346 0.23015586 -0.18401974 -0.11697002 0.124981776 0.14820522 0.27618563 -0.04181812 -0.113901965 -0.009699008 0.3347717 0.019119801 -0.23122233 0.046083126 -0.019014122 -0.18919545 -0.035755232 -0.3319628 0.2030143 0.1547306; -0.081128895 0.13469566 0.08533799 0.12193733 -0.04390153 -0.0032420545 0.25022867 0.04451063 0.03738585 0.053689145 -0.10176001 0.09015714 -0.06139167 -0.051439356 0.20750639 -0.13350351 -0.06076338 0.15750797 0.29173827 -0.0917776 -0.013574857 0.106353 -0.0015392989 -0.28410193 -0.27882472 -0.035430152 -0.015980897 -0.108299 -0.08228014 -0.2566559 0.17935532 -0.006208051 0.03861458 0.016634613 0.26441053 0.19636044 0.039622072 -0.12785098 0.21783312 0.25743142 -0.23645489 -0.044209618 0.043062612 0.023967203 -0.061742168 0.21900094 -0.027600236 0.08917396 0.26879337 0.12214036 0.18264894 0.1045198 -0.027787467 -0.14689852 -0.0056191366 -0.0006380626 -0.13367148 0.04561203 0.1268129 0.0682166 -0.08012516 0.055086866 -0.12052097 -0.04482818 0.0323379 -0.061099537 0.027797833 0.20381823 -0.03860046 -0.07727037 0.00016626877 0.18923022 -0.0055443645 0.10148216 0.12524942 -0.16427661 0.08204126 0.03468223 0.13815689 0.015387167; 0.12196364 -0.13537106 -0.10179118 0.26899529 0.08889924 0.12565263 -0.09671262 -0.114513084 -0.12927864 -0.003163717 -0.34673238 -0.04667197 0.08212452 0.19297895 0.18856329 0.15425718 -0.2330385 0.23726216 0.10095754 -0.055844452 -0.021989223 0.25658807 -0.0020249079 -0.29416776 0.097740464 -0.13541839 0.1121338 -0.053817164 0.2482381 0.18968025 -0.057646 0.033930525 -0.34251165 -0.3393186 0.16129977 -0.020196069 0.31093505 -0.18555796 -0.00897167 0.1696947 -0.1072137 0.108817615 -0.031487744 0.34910783 -0.018272508 -0.16344878 -0.11724135 -0.16219606 0.07660164 0.06574431 0.016428096 -0.1517709 0.25290376 -0.15758052 -0.25102672 0.19332077 0.11528375 0.09249273 0.12639612 -0.32304764 -0.11147219 0.07902927 0.17774798 0.04171696 0.044111524 0.06868013 0.07128087 -0.045592755 -0.2528487 0.00568221 0.2137158 0.17795515 -0.24247123 -0.014948155 0.2963402 -0.19748382 0.14028724 -0.30404714 0.18274726 0.006791769; -0.23392417 0.3088853 0.23888332 -0.23859517 -0.035939116 0.16667607 0.044233702 0.03043458 0.21570712 0.29792783 -0.047580026 0.15673164 -0.108237796 -0.1601507 0.20250046 0.2218229 0.011335641 0.1139217 0.11984733 0.2347648 -0.09569721 -0.21382396 -0.0057298667 -0.04279177 -0.29097492 0.26927668 0.07942952 0.061649762 -0.08073411 -0.34918824 -0.014060806 0.19717468 0.20979026 0.2474392 -0.17619856 0.046377603 0.012235677 -0.03346119 -0.07881009 -0.031302355 0.1806495 -0.053147066 0.10310787 -0.08552735 0.11201382 0.21052384 0.22656283 -0.28992304 -0.015129016 -0.12660079 0.19312404 -0.119658396 -0.052086394 0.14205213 0.15662922 -0.04860196 -0.2337958 -0.13097213 0.21803355 -0.14538594 0.22165397 0.04980461 0.11417079 0.19737175 -0.017096588 -0.13718739 -0.27914038 -0.22298813 0.04575982 -0.230576 -0.25632873 -0.19432944 0.26306823 0.14460349 -0.2362617 0.03631302 -0.07588717 0.0601601 0.046115994 -0.051351335; -0.08834014 -0.060313392 0.2549776 -0.18505271 0.0014317544 -0.23025197 0.3114355 -0.022455197 -0.05875257 0.31598568 0.06496056 -0.09154707 -0.14540115 -0.3000118 -0.10603535 0.18516472 0.0031812335 -0.1816267 0.0055013862 -0.15766776 0.30059722 0.0064833895 0.16605374 0.3689359 0.0038116549 0.2673823 -0.057903223 0.17143466 -0.26798627 0.044490658 0.02751045 -0.30321512 0.16633351 0.2652447 -0.29083028 0.13148582 -0.010036717 0.21562454 -0.22446425 -0.17436713 0.082418784 -0.3346383 0.1337181 -0.27797315 0.29304636 -0.13035007 0.23413447 -0.06615523 0.2510787 0.040770948 0.22120455 0.19775634 0.007921694 0.18122992 0.23688233 0.015927605 0.041580077 -0.23112871 -0.26369068 0.4038604 0.2551891 0.0014418149 -0.082539245 0.1982703 -0.14882104 -0.12743267 -0.30924958 -0.3030565 -0.00046824652 0.0991044 0.19906458 -0.15737103 -0.14081569 0.2251805 -0.3437239 0.2627793 0.063522145 0.1924703 -0.2345991 -0.062049836; 0.2502856 -0.06469463 0.02953918 0.16564952 0.27616516 0.05009845 -0.08168585 -0.043961257 -0.19272597 -0.009089346 -0.1931995 0.12936434 0.028194578 0.10108193 -0.027285317 -0.14157848 -0.06749884 0.24583872 -0.12411913 -0.131814 -0.26954734 0.14682873 -0.2809373 -0.21188717 0.033713374 -0.3025194 0.2799587 -0.32905963 0.048089776 0.12581113 0.029798802 0.07277875 -0.12757741 -0.09083694 0.1064347 0.0096958745 0.2303678 -0.10002198 -0.05584397 0.005840167 -0.26736835 0.00823827 -0.2425433 0.26705933 -0.098311156 -0.16595897 -0.05830704 0.01266634 -0.24208814 0.020996943 -0.01340323 -0.22737595 -0.01415339 -0.056792304 -0.1239171 0.25835878 0.28569505 -0.008060968 0.07544194 -0.005511715 -0.012281037 0.03732675 0.19678606 -0.24299943 0.15445001 0.12814261 -0.028470742 0.14537235 -0.038876507 -0.057640653 0.27492827 0.17324926 -0.14845519 -0.040901046 0.16533698 -0.011386355 -0.043016113 0.0119512025 -0.035461083 0.043601435; -0.059697684 0.2364199 0.27142325 -0.22463313 -0.3272276 0.1833092 0.27303424 0.011065628 0.26925707 0.009916443 0.0042145187 0.037907768 -0.072429195 0.042939182 -0.03424707 -0.1627382 0.09327271 0.2104385 0.23192185 0.008811343 -0.0307619 0.06972457 0.08147003 0.13311481 -0.044916824 0.022323176 -0.05302831 0.2671861 -0.09045076 -0.2348355 0.18581013 0.20839581 0.070803754 -0.012756928 0.07250255 0.21695639 0.010713671 -0.025734842 0.15958326 -0.05371356 -0.17216271 0.020849863 0.30411962 -0.17311567 0.19901723 0.14929415 0.12808017 -0.07968025 0.082907274 0.007918147 0.28541452 0.11972615 -0.27870655 -0.23388177 0.052690215 -0.009479363 0.010619936 -0.09305144 0.0016710913 0.14339311 -0.1427369 -0.24677648 -0.06912113 0.08033996 0.16184074 -0.1462944 0.015100195 0.05959466 0.105854794 -0.15314364 -0.124562696 -0.13313983 0.20329873 0.301111 0.01952875 0.06669126 0.07845221 0.26413143 -0.25306487 0.0521503; 0.07134429 -0.3157977 -0.14226982 0.14424361 0.18227366 -0.30425507 -0.10077163 -0.022630675 -0.08409497 -0.13838415 -0.08493553 -0.17112601 0.21632159 -0.09554153 0.032658633 0.19252075 -0.21940164 0.1547012 -0.3173097 -0.34710932 -0.1384466 0.107735686 -0.21094263 -0.09529475 0.22929828 0.06219095 0.20939861 0.054532133 0.16156222 0.11834065 -0.0030526528 0.014821281 0.0120785115 0.02682067 0.19917955 -0.17408822 0.22354925 -0.0046436377 -0.048009116 0.19508848 0.10203755 0.054255962 -0.2971989 0.233512 0.027271397 -0.2598192 -0.10239134 0.33664888 -0.18563725 0.22765696 0.049855966 -0.05038988 0.21530357 -0.24728209 -0.07953125 0.0015476826 0.2769412 0.17880535 0.074879475 -0.043620978 0.0062792436 0.1578741 0.16448806 -0.27191037 0.12820865 -0.027284663 0.27055016 -0.023270125 -0.28606626 0.15255165 0.109780304 0.089741945 -0.17985183 -0.17294063 0.22089505 -0.02503614 0.060178276 -0.024570623 -0.039862316 -0.0009924539; -0.02415759 -0.109991364 -0.16564137 0.061670862 -0.03802926 -0.2797213 -0.21079047 0.062384177 -0.16443554 0.02554055 -0.0028440473 -0.18324178 -0.025663285 0.11166921 0.25730622 0.08700154 -0.07501289 0.065070815 -0.05597085 -0.050766062 -0.026209397 -0.052164026 -0.02596485 -0.043477897 0.22230585 -0.15499614 0.31112567 0.042467758 0.09408501 0.1329069 -0.27743766 0.12141767 -0.2218097 -0.3153086 0.21284574 0.0036051322 -0.05170423 -0.3028546 0.3105211 -0.00038592343 -0.026469609 0.27987865 -0.26606387 0.26019335 -0.096930824 -0.06431033 -0.05379789 0.071894035 -0.23142843 0.30205333 0.005628843 -0.044662304 0.030002799 -0.12660946 -0.2564687 0.032732237 0.11273643 0.18619825 0.10277074 -0.04010768 0.036191244 0.15239966 -0.02645522 -0.23487385 0.0027884208 0.04543126 -0.041040502 0.23454954 -0.120316975 0.2770783 -0.036027268 0.20442574 -0.30532634 -0.16169026 0.11093774 -0.28095248 0.03724433 0.0518874 0.2956975 0.09805273; -0.021507747 -0.23800015 -0.016599191 0.09381939 0.18987378 -0.32951435 -0.012011458 0.14427061 -0.27455983 -0.01911176 -0.24294062 -0.105818756 0.13379009 0.1006688 0.25059497 0.25196528 -0.26135176 0.026888793 -0.11924628 -0.26463854 -0.17009628 0.10159825 -0.08962054 -0.30830288 0.15330036 0.011467699 0.22376288 -0.094669744 0.14741145 0.022645226 -0.3211953 0.27008414 -0.07109057 -0.31756186 0.059241503 -0.28268087 0.13159795 0.02531392 0.08452451 -0.011410794 -0.2181024 0.14450406 -0.3001513 0.25298706 -0.27304363 -0.004706353 0.050435714 0.18582556 -0.049213745 0.18951705 -0.13132362 -0.0504145 0.13087831 0.029001513 -0.0996875 0.13558315 0.3195938 -0.020557145 0.2738087 0.053503197 -0.13969894 0.16116387 0.2068491 -0.2944711 0.25577852 0.28030086 0.02866235 0.048529673 -0.0328782 0.08276131 0.16982546 0.08380942 -0.203837 -0.24395481 0.078109674 -0.015205446 -0.07245778 -0.23137803 -0.016092367 0.04030626; -0.00039103554 0.07246013 0.22033134 0.10086479 -0.27353728 0.12684351 0.21605997 0.087898836 0.30500704 0.109717526 0.11304754 0.20632693 0.04559824 -0.0064826137 -0.12631397 -0.2871156 -0.012588558 0.001748494 0.24659322 0.11988882 0.18943314 -0.17642222 -0.08698762 0.18351348 -0.08469838 0.23591968 -0.082199305 -0.023323521 0.09249618 -0.20147935 0.057425622 0.16387774 0.047589276 0.23324852 0.15199511 0.19030589 -0.23167871 0.03768221 0.0867223 0.14273795 -0.15040006 -0.043799937 0.36529756 0.12685366 -0.060872056 0.047790047 0.25059423 -0.10327114 0.25574887 -0.050955214 0.26127347 0.0021860776 -0.056007437 0.109972596 -0.016019203 -0.23184188 -0.29362234 -0.23521478 0.23270413 0.16896546 -0.16825062 0.019456547 0.10092688 0.023347197 0.031764302 -0.07436684 -0.17308919 -0.21102348 0.19547448 -0.117761426 0.009730064 0.084948264 -0.0240388 0.33097053 0.053039305 0.22998258 0.01163559 0.083489515 -0.17927606 -0.26903155; -0.1524909 -0.047627825 0.00656743 -0.016179712 0.042682562 0.18842207 0.18319675 0.18030062 0.26393938 0.20282012 0.21061283 0.08687951 -0.17352933 -0.13321114 -0.019135704 -0.10980348 0.057855636 -0.2677416 0.297447 -0.045869447 0.067523725 -0.14711803 0.16267312 0.0841479 -0.20191295 0.020005014 -0.1042836 0.12243872 -0.11277607 0.040460378 0.039722096 -0.05343671 0.048900235 0.127657 -0.3435821 0.09498423 -0.1565994 0.101929486 -0.31129077 -0.3376898 0.28518552 -0.088819854 0.18137802 -0.19323537 0.09763314 0.21709196 0.21952786 -0.28183904 -0.012058065 -0.29790205 0.32518739 -0.03231939 -0.28521812 0.32896304 0.19248934 -0.26844516 -0.13096441 -0.28317332 -0.29551706 0.13432187 0.095527835 -0.12749344 -0.22864051 0.1915901 -0.11029893 -0.044777006 -0.33571985 -0.11045722 0.1275812 -0.1355405 -0.2864084 -0.21085417 -0.067120105 0.091996945 -0.024996545 0.22341846 0.06364778 0.021392638 0.04078347 -0.20778996; 0.20358902 -0.1307434 -0.09733714 -0.21455662 0.14041162 0.054417517 -0.14898895 -0.025182854 -0.15143056 0.044208363 0.051104758 -0.18386798 0.059717588 -0.016908452 -0.036300916 0.28392163 -0.061410237 0.038254544 -0.020313619 0.034268636 0.1727794 -0.12833919 -0.027661674 -0.03125829 0.24588259 -0.24494444 -0.051063642 0.12339449 -0.18132955 0.11244632 -0.18257964 -0.21434338 0.00086999347 -0.08058647 -0.085242964 0.04522461 -0.063462794 0.0118219005 -0.12091537 -0.079101294 -0.12968989 -0.13896663 -0.2939139 -0.020493016 0.26741546 0.11882696 0.1520405 0.2950193 0.03619172 0.09599603 0.003679288 0.09353597 -0.012115275 -0.026879473 0.15240031 0.0009182424 -0.14351812 -0.06497635 0.04207976 0.12130341 -0.010152238 -0.0380179 -0.21788633 -0.15631467 -0.2008294 -0.13648093 -0.123034 -0.075780086 -0.11895578 0.24057318 0.19843456 -0.184548 -0.043512087 -0.19073747 0.008943912 0.18224996 0.061586868 -0.22961971 0.09459366 0.06627925; 0.33068287 -0.30627984 -0.25422356 -0.06897183 0.1019787 -0.2772481 -0.26050863 -0.06252324 -0.35552508 0.04334905 -0.18578269 -0.25178614 0.20515637 0.053073224 -0.07521961 0.11299685 0.081730165 -0.120682456 -0.23708908 -0.30389726 -0.012373024 0.059795693 0.11255393 -0.2553712 -0.02654231 0.028864933 0.075324945 0.052237034 -0.013549451 0.17622301 0.05972211 0.069408074 -0.19170475 -0.09580826 0.07602437 -0.24052405 0.22470073 -0.14192183 -0.09123416 -0.12609656 -0.03780529 0.036328394 -0.24442257 -0.06315784 0.1469397 -0.22347121 -0.1282217 0.042994227 -0.343229 -0.1025827 0.042365715 0.09584485 0.24867779 -0.0008785735 -0.2647684 0.0880504 0.2824478 0.008705651 0.10231717 0.07109333 0.12584138 -0.05843845 0.15257846 -0.3358642 -0.11344728 0.252567 -0.04187532 0.16842647 -0.058040958 0.2191895 0.09403046 0.15454036 -0.18439662 -0.3189167 -0.0063868156 -0.103179336 -0.16057667 0.029685471 0.037736155 -0.02409611; 0.27339095 -0.18286778 -0.019779412 -0.057960793 0.0138076935 -0.32484972 -0.26681465 -0.0972156 -0.2743733 -0.02334607 -0.07181671 -0.04863835 -0.03247554 0.2070913 -0.21143928 0.195295 0.07550653 0.11136508 0.0055793156 -0.27089953 -0.14717752 0.027882187 -0.15156035 0.055220593 0.2063822 -0.009075489 0.026251784 0.017712278 0.2249409 0.22655284 -0.10848845 -0.065663755 -0.020460155 -0.20426574 0.13731219 -0.28797668 -0.023080222 -0.061069436 -0.08483419 -0.018407434 0.009160215 -0.05895194 -0.1289387 0.054132257 -0.24923663 -0.1339478 -0.11705798 0.0520491 0.024760531 -0.034515593 -0.076371044 -0.0036364305 0.0176451 -0.1958895 -0.17347994 0.0061524673 0.10881977 0.16252065 0.07610858 0.009095147 -0.14018025 -0.008436497 -0.1276603 0.017846497 -0.0042273677 0.109172896 0.035139658 0.09131341 -0.00038013083 0.052933533 0.23042612 0.12879257 -0.05474059 -0.09050439 -0.08763701 -0.051582556 0.06016813 -0.16688082 -0.033697367 0.08212861; -0.27136782 0.25733432 -0.11715451 -0.16811977 -0.2286327 -0.002948875 0.044050105 0.08571777 0.14582582 0.18778792 -0.01062612 0.1194492 0.13338548 -0.12942487 -0.085387506 0.06657764 0.10735494 -0.032719526 0.20168783 -0.010663941 -0.21418877 0.045282297 -0.10214071 0.14030363 -0.27661917 0.22255456 0.11217605 -0.045210063 -0.16432811 -0.24288665 -0.06282634 0.13993649 -0.10829995 -0.11148975 0.10018255 0.27581212 0.06907303 0.22387952 -0.153491 0.0012627788 -0.19976108 -0.1016206 0.2364131 0.026410392 0.00019903129 0.24080297 0.20860529 -0.09364406 0.11388481 -0.15790205 0.07813635 -0.003984702 -0.20322685 -0.17945965 0.23593055 0.13284327 -0.19937775 0.1421389 0.22013913 0.020628968 -0.12562507 -0.17442845 0.21563275 0.045274142 0.09791515 -0.09108235 -0.1718731 -0.12935606 0.0034249255 -0.18738742 -0.13559754 -0.068279244 0.27222788 0.34382308 0.097720444 -0.020669056 -0.13627739 0.07563913 0.026064727 -0.14113979; 0.19399273 -0.122196905 0.16464476 0.028663982 0.0987699 -0.11187454 -0.27855492 0.0011018296 -0.33385777 -0.021937298 0.09413685 -0.11356052 -0.09682958 -0.22660246 -0.19264022 0.25078002 -0.073193274 -0.2696303 -0.115560524 0.03930784 0.060295533 -0.09255889 0.031675104 -0.032764036 0.18707283 -0.115480065 0.058560953 0.21638815 -0.14633012 0.015810564 -0.26426545 -0.2509394 -0.029758122 -0.023210024 -0.22358114 -0.10797401 0.055739872 0.22317663 0.10677446 -0.16473417 0.005648987 -0.1540712 -0.21698491 -0.09264706 -0.10789669 -0.011042072 0.08334581 -0.021822575 0.05402857 0.043289788 -0.20798528 -0.115592495 -0.036440376 -0.06261968 -0.063251615 -0.09047296 0.0073495186 -0.15669979 -0.050779197 0.24015284 0.1447056 -0.11318249 -0.20116232 -0.26698267 -0.12974513 0.046176378 -0.1219592 0.044018935 0.065510504 0.09917511 0.3451667 -0.14616963 -0.047669113 -0.128415 -0.035011638 0.1911503 -0.1453661 -0.18745942 -0.10933543 0.12879755; 0.26434338 -0.18324545 0.03404461 0.19422147 0.05231708 0.03786629 -0.056239538 -0.12929074 -0.13890792 -0.24162194 0.055927265 0.07364176 0.16140932 0.13159515 0.17905186 -0.07644839 -0.104541525 0.2763398 -0.08047301 -0.093711376 -0.27432355 0.12495175 -0.14194342 -0.08685242 -0.039559346 -0.25834385 0.25947982 -0.29226425 0.20470825 0.22761549 -0.1987707 0.19127688 -0.11494744 -0.20762932 0.27706474 -0.20673464 -0.00843677 -0.092016295 -0.03794982 0.031679153 -0.26959905 0.24620022 -0.1729633 0.044391155 -0.10030728 -0.18204749 -0.28037655 0.13203992 -0.0432554 0.26540115 -0.12072932 -0.2282838 0.08632384 -0.217848 -0.25836045 -0.029589916 0.01168776 0.043858435 0.15343662 -0.108544625 -0.099331565 0.28722316 -0.043741897 -0.08411228 0.09348119 0.062572636 -0.049077023 0.06632847 0.035676852 0.0065568276 -0.018208673 0.1591691 -0.30070546 -0.049569 0.14726155 -0.055255696 0.06950753 -0.059686534 0.32021257 0.07449177; -0.0047295997 -0.34354326 0.09459922 0.13442576 0.2914858 -0.061413024 -0.018257722 -0.041130997 -0.3027351 0.024255073 -0.069556616 0.017744204 0.21543783 -0.018510116 0.030659286 0.04949926 0.058191806 -0.21712966 -0.13495667 -0.040545467 0.14094578 0.10597381 0.16213909 0.0648348 0.032685988 0.001646158 0.04531049 -0.117347084 -0.016149972 -0.023802113 -0.14993246 0.03214076 0.013922868 -0.10928496 0.120262414 -0.26776993 0.20982844 0.010678245 0.056587264 0.103854954 0.031245705 -0.049571898 0.010036893 0.012361527 0.125671 -0.32707483 -0.018558037 0.012225482 -0.028093759 0.17289372 -0.09595501 -0.18337381 0.047068406 -0.112524316 -0.043471865 0.123087086 0.10115968 0.18274565 -0.21490039 -0.04749329 0.1518244 -0.058872063 -0.023725186 0.0047649234 -0.10388101 0.10372122 0.021168282 0.14914326 -0.26870418 -0.0034662155 0.18981299 0.009807282 -0.25097317 0.004225693 0.04541407 -0.12631273 0.116269864 -0.20737237 0.17880197 0.08894217; 0.27996823 -0.09211136 -0.18953282 0.022488281 0.069209315 -0.033324383 -0.024662862 0.16586466 -0.012921135 -0.21317688 -0.1016474 0.043071274 0.2632777 -0.001493945 0.08692973 0.2745142 -0.020941896 0.20772567 0.021601005 -0.13677616 -0.11723355 0.093405806 -0.21508715 0.07763235 0.24642448 -0.16416563 0.20529845 -0.02712298 0.07619614 0.05509044 -0.11319488 0.20527881 6.808718f-5 -0.16750108 -0.09426664 0.041863702 0.2924403 0.054097902 0.08357513 -0.00732999 0.08152795 0.26231873 -0.010038046 -0.09860363 -0.17877783 -0.20375508 -0.091730505 -7.816937f-5 -0.054433767 0.24564674 -0.11782922 -0.14779508 0.3318184 0.01729324 0.07897856 0.18191732 0.233822 -0.05110867 0.1417512 -0.030135186 -0.15078597 0.22236335 0.030202061 -0.1279947 -0.04818918 0.16043976 0.2712688 0.037783366 -0.039842937 0.03603215 0.11133701 0.0636934 -0.2905727 -0.26010567 0.1416002 -0.14452924 0.10281205 -0.20501064 0.25687286 0.06801924; 0.16288905 -0.0050146664 -0.2014905 -0.13635273 0.31327006 -0.18138953 -0.20018114 -0.08919346 -0.030198088 -0.040901184 0.11577456 -0.050153214 0.035941664 -0.13154931 -0.17002818 0.15862925 -0.042803288 0.08694644 -0.33905733 0.011580658 -0.12255075 0.084793955 0.018899145 -0.05558251 0.0963704 -0.20463693 0.17034806 -0.11800476 0.2300862 -0.0059365104 -0.013999519 -0.1539108 -0.1688582 0.08309441 0.07962228 -0.30783832 0.080071814 -0.2163903 -0.15401244 -0.06992686 0.08547522 -0.08579622 -0.14621897 0.16044298 0.066877484 0.02841343 0.08584615 0.086096466 -0.20470989 0.12630521 -0.17414825 -0.24308653 0.10347534 0.2216143 -0.23106782 0.10806887 0.13824111 0.036678802 -0.16658777 0.09041355 0.008949871 0.30405065 -0.16930111 -0.044048794 0.16980639 0.041347254 0.26038122 0.20283657 -0.0066586793 0.25770947 0.26345167 0.20533916 -0.33718374 -0.34964794 -0.1050042 0.10979111 -0.0016118959 -0.10173608 0.012052288 0.20422134; 0.22773884 -0.01856997 -0.21129689 0.2633795 0.05629835 -0.057181425 -0.18143624 -0.09873522 -0.29978448 -0.09436119 -0.023086982 -0.16817789 -0.052003168 0.24436124 -0.12579362 0.05371479 -0.13047363 -0.0006301745 -0.1080114 -0.034095682 0.059486154 0.008244014 -0.12525535 -0.18919441 0.15318137 -0.083169244 0.25867018 -0.19693206 -0.0023830596 0.049496427 -0.07387491 -0.076432705 0.008473119 -0.069421746 0.24963345 -0.18181871 0.18622364 0.05102301 0.193755 0.13460192 -0.2256362 -0.04481996 -0.28218457 0.1198955 -0.1134325 -0.3121247 -0.14547142 0.2803956 -0.023097413 0.15923482 -0.27755132 -0.0031154105 0.32023937 0.032594476 -0.24380285 0.036511377 -0.06719682 0.030008925 -0.13475825 0.027274556 -0.1782123 0.0961584 -0.054997522 -0.10348176 0.090454556 0.18462315 0.1838469 0.11310552 -0.01008363 0.25306994 -0.04365068 0.20848873 -0.15448382 -0.25302282 0.15141273 -0.22380777 -0.011970618 -0.24530517 0.06460636 0.28322983; -0.06871434 0.13655058 -0.06684927 -0.05014381 -0.21199378 0.16057467 0.18382107 0.21540307 0.27667642 0.07538003 0.071173154 -0.12785694 -0.16391252 -0.32804462 -0.04100764 0.083666116 0.30730587 -0.16789545 -0.013646553 0.032376464 0.1415994 -0.046363316 0.09170082 -0.013002705 -0.1716821 0.31570274 -0.05896284 -0.005464581 -0.20047693 -0.29726484 0.08423504 -0.23317222 0.055042163 0.30391243 -0.03709902 -0.028830286 -0.26880023 -0.017320568 -0.2704962 0.018692067 0.28482494 -0.21308944 0.022845862 -0.008578003 0.14590785 0.04260447 0.07415872 -0.07313988 0.1365905 -0.2533692 0.29592723 0.14134492 -0.1136517 0.023061484 -0.0580434 -0.08674205 -0.018426072 -0.013486109 -0.014711967 0.2125922 -0.0021010367 -0.23908454 -0.13017812 0.16555452 -0.19359812 -0.12068358 -0.29272223 -0.22137147 -0.026229255 0.001708156 -0.28522232 -0.20606567 -0.0066007944 0.08109396 -0.076757975 -0.025805319 -0.23072061 0.31719378 0.005887045 -0.055460766; -0.07265697 0.2772659 -0.00075178413 0.06508861 -0.22800687 0.04986743 0.076571114 -0.25612697 0.23744705 0.013120362 -0.13477863 0.16738835 -0.25908542 0.06529974 -0.12063636 -0.10412987 0.28143087 0.013870423 0.20270155 0.2645573 -0.07652849 0.13097604 0.23564538 0.25071946 -0.005384003 0.13815732 -0.27434012 -0.071725056 0.08992488 -0.10973803 0.05541306 -0.096144214 0.042322345 0.07755227 -0.068678945 0.059063487 -0.18371549 0.297233 0.17222871 0.026858296 0.19288954 0.0049291705 0.08384354 0.05946877 0.18892367 0.0024619626 -0.111283794 -0.13435835 0.103662476 -0.08846589 -0.06263335 -0.01936537 -0.07155535 -0.10206762 0.10909158 0.094980225 -0.30197194 -0.2345911 0.052758306 -0.08987151 0.2043889 -0.27422297 -0.1485424 0.17475957 0.05394951 -0.27057418 -0.23633336 -0.11602758 0.15434998 -0.09308476 -0.24760813 -0.13467535 0.2704871 0.22514103 -0.052688174 -0.012389839 0.109392785 0.07789467 -0.16038243 -0.19461505; 0.2861286 -0.19484681 0.17683703 -0.25996605 0.14747445 -0.18837279 -0.12107314 0.05038175 0.028289296 -0.077503234 -0.10530845 0.08455952 0.06696104 -0.15280786 0.017565342 0.1760599 0.102004506 -0.0119620245 -0.05376933 -0.09789918 0.11178549 -0.20150283 0.21215802 -0.025603786 0.10200615 -0.18046495 -0.058982465 0.2184999 -0.08705466 0.17565687 -0.10176429 -0.18221006 0.049331136 0.1686521 -0.156747 -0.14773536 0.08961444 0.18598957 0.07332492 0.0006630257 -0.0021583063 -0.13492179 -0.08181403 0.042051632 -0.10692988 -0.12057643 0.11811569 0.06754128 -0.0060354793 0.0357953 -0.19977385 -0.11508669 0.06577105 0.029767474 -0.03970066 -0.049108155 0.16256787 -0.16525394 -0.0950784 -0.009810448 -0.03597952 0.06491246 0.025099693 -0.15467131 0.08330744 -0.0943779 -0.117364295 0.008615162 0.14239825 0.19463317 0.22323394 0.14048359 -0.12572943 -0.21833377 -0.07192337 0.16369353 0.043121763 -0.015666956 0.052076556 -0.15815552; -0.35793576 0.27555862 0.20335579 -0.0960261 -0.23109855 0.6143516 0.032641377 0.13714953 0.19412938 0.16866747 0.06980951 0.0247296 -0.2043367 0.02253624 0.0609692 -1.3986236 0.18091111 0.08522451 0.18319434 -0.06502796 0.114682 -0.018080318 0.114089414 -0.107728824 -0.80061746 0.038723867 -0.144463 0.043793175 -0.17273697 -0.25878608 0.29119992 0.18888327 0.2338076 -0.058080662 0.028415736 -0.022623159 -0.0043592975 -0.057290092 -0.059071936 -0.18209583 0.06905301 -0.13791506 0.27640203 0.06299162 -0.015504483 0.09864974 -0.08396875 -0.035943385 0.2923604 -0.17517594 0.3643946 0.27965242 0.10813876 0.13074407 0.019486606 -0.13909143 -0.076270424 -0.051117137 0.14902005 -0.06344408 0.054562226 -0.13455522 0.07831335 -0.019054526 0.07963562 -0.42405713 0.04921302 -0.10539584 0.22839276 -0.43058077 -0.4283511 -0.1476205 0.19351979 0.3544619 0.034482405 -0.047947392 -0.21567811 0.02268769 -0.30597758 -0.07619309; -0.11881258 0.24200456 0.07933155 -0.21406303 -0.04932493 -0.047910612 0.08892376 -0.10572275 0.12585877 0.29067913 0.3007339 -0.06332276 -0.31158125 -0.009425813 -0.22063719 0.1492505 0.045651004 -0.2946209 0.27377746 0.20768571 0.2989262 -0.18742637 0.25325653 0.097963616 0.08239869 0.12523901 -0.2859117 0.011833895 -0.2566269 -0.30254218 0.17991811 -0.17324783 0.2678189 0.1215917 -0.24081366 0.09188578 -0.0784385 0.22395219 0.042852975 -0.31108254 0.21110593 -0.10904225 0.06832464 -0.18585469 0.119501345 0.07768322 0.29432428 -0.15913498 -0.028356193 -0.14142409 -0.020455904 0.17759995 -0.1516848 0.29769957 0.096158125 -0.21585503 -0.17728041 0.05174915 -0.30269068 0.38742507 0.064854614 -0.034623723 -0.059388738 0.24014597 -0.29405326 -0.18879372 0.021576256 0.019002926 0.09116463 -0.13834059 -0.14402899 -0.27190617 0.15095876 0.20051299 -0.027692389 0.24355675 -0.1975769 0.20265149 -0.19254164 -0.30278876; 0.19890563 -0.07404089 0.07292583 0.18262234 -0.036101867 -0.2462055 0.037042134 0.13969867 -0.23844866 0.041464347 -0.12033935 -0.1524238 0.23595327 0.013591812 0.082017906 0.012585965 -0.1724831 0.14131263 -0.031358875 -0.31446996 -0.19311355 0.015906852 -0.24633564 -0.26173136 0.044630878 0.055383492 -0.07070702 -0.1671454 0.18433358 0.286088 -0.22936618 -0.03837648 -0.1017237 -0.226942 0.006888541 -0.17222916 0.27073243 -0.2248617 0.18683484 0.140272 -0.0055185133 0.2327465 -0.30408806 0.21774054 -0.19292772 -0.3324764 -0.2599526 0.19081077 -0.16059026 -0.07436837 -0.2350189 -0.14563406 0.012601176 -0.06153887 -0.13680154 -0.015744075 0.039477788 0.032817777 -0.22622083 -0.25009093 0.062158056 -0.07575457 0.03290539 -0.11283436 0.17928107 -0.026516177 -0.077619 0.28302202 -0.23886527 0.0016620423 0.15738131 0.025353173 -0.12983863 -0.0046871337 0.28158543 -0.25615826 -0.059390794 -0.10723723 0.093512684 0.21388133; 0.058989525 -0.08337044 -0.2940582 0.079850845 -0.097496636 0.22441426 -0.04776866 -0.13288297 -0.0010729888 -0.23783813 -0.10827822 -0.069126174 0.15910636 0.31097022 0.05250755 -0.1413741 -0.15215889 0.28085926 0.072453015 0.044706706 -0.2946604 0.16904569 -0.024192328 -0.116958365 -0.3062446 -0.30528715 0.37437406 -0.1318773 0.20325026 -0.12517726 0.069842234 0.0798687 -0.25816315 -0.2745571 0.08947255 -0.27911726 0.031644735 -0.32712048 0.19045256 0.2535599 -0.13385935 0.018900309 0.036943812 0.32325438 -0.106733255 -0.08858155 -0.15721431 -0.1660926 -0.16866124 0.19891725 -0.05597187 -0.07538745 0.29484463 -0.06800961 -0.30165836 0.14764497 -0.025521418 -0.052842803 0.0025677697 -0.096365795 -0.40219966 0.105665684 0.13219535 -0.23964936 0.29937246 0.0021014023 0.34323564 0.08390092 -0.029804423 0.18693493 -0.0023615048 0.09100387 0.061733786 -0.21638547 0.3217766 -0.25037634 0.23810597 -0.15544885 0.21341543 0.37007543; 0.00057540066 0.09083591 0.24718362 -0.03431272 -0.107762374 -0.078637995 0.041157592 0.21339685 0.010406692 0.13104756 0.26863128 -0.0149630755 -0.2880403 -0.1871944 0.04066808 -0.38057828 0.09343916 -0.201094 0.06271207 0.1877838 0.21379559 -0.3213628 0.16506302 0.21602371 -0.076442584 0.23633203 -0.052452404 0.22791652 -0.026223978 -0.11036925 0.0013106663 -0.1909332 0.007365656 0.25006008 -0.3244962 0.006483438 -0.3368615 0.23740707 -0.14154726 -0.35423324 0.005261195 -0.2599056 0.031873 -0.36149856 -0.011802895 0.035941064 0.075248286 -0.24644242 0.007034909 -0.23151903 0.15619521 0.0081915585 -0.06518133 0.14890367 0.23735258 -0.16527987 -0.21336843 -0.17417993 -0.27874333 0.08065112 0.09759267 -0.048566762 -0.06737616 -0.0037651411 -0.1304111 -0.24866034 -0.15398689 -0.143731 0.23481342 -0.07488597 -0.20939109 -0.19092844 0.2411548 0.07069831 0.01703668 0.33951947 -0.27442616 0.14664061 -0.30023268 -0.08548607; -3.5878493f-5 0.058145266 0.23991571 -0.27154845 -0.18358149 0.05054554 0.044239465 0.07297373 -0.015228032 0.28202015 -0.07701493 -0.06542043 -0.22562492 -0.041009575 -0.13328408 0.010626171 0.2809316 0.027306259 0.08479837 0.09466798 0.006796267 -0.080466375 0.23908705 0.28238487 -0.060615133 0.22540627 -0.0064858855 0.30137986 0.089693345 -0.18690926 0.3205921 0.13618702 0.26421085 0.06646137 -0.10529604 0.25464353 -0.15006192 0.29043102 0.027692357 0.0077317352 0.07589978 0.03979745 -0.030515088 0.043228257 0.04422579 0.2742453 -0.039008483 -0.006017405 0.29559556 -0.13143957 0.21745105 0.2323586 -0.050538354 -0.14054786 0.017716164 -0.11934988 -0.023758251 0.070949584 0.11708345 -0.013883001 0.11616569 -0.2844302 -0.025279429 0.029154906 -0.1114426 -0.14664714 0.08968153 -0.22888775 0.20230012 -0.2797426 0.03730544 -0.08452269 0.03728836 0.28768897 -0.0029851305 -0.0795697 0.1315804 0.116656125 0.03484807 -0.29896834; 0.13752761 0.00602034 -0.23031242 0.029844781 0.028811567 -0.0841732 -0.095050775 0.2029789 -0.012104039 -0.2909924 -0.21933559 -0.18791507 -0.0037370094 -0.031328928 -0.010475048 0.03295429 0.014936878 0.11052994 -0.18370369 -0.07714348 0.10931286 0.22191724 0.01461764 -0.03610548 0.29449022 -0.084065594 0.23513062 -0.281785 0.19190998 0.0052078622 -0.17842858 -0.07681846 -0.02175936 -0.07377834 0.081732966 -0.038049646 0.014598698 -0.043890852 -0.00714227 0.15213944 0.06689889 0.06295625 0.008220096 0.25055718 -0.032920595 -0.35960594 0.0790668 0.18471691 -0.026588995 0.08784578 -0.16950001 -0.08425688 0.11572619 0.052492086 -0.094987564 0.055404875 0.18765599 -0.0016600081 -0.16162358 0.11756655 0.02996675 0.095887624 0.19903292 -0.101913504 0.13664204 -0.00026507568 0.114555165 0.26233408 -0.12954576 -0.014973761 0.19426237 -0.010589198 -0.15655407 -0.2663457 0.25596 -0.14046767 -0.1320369 -0.13626234 0.06401469 0.07914499; -0.16295041 0.29192042 0.26170808 -0.06700859 -0.21240696 0.16258371 0.16880915 -0.18319869 0.037092667 0.08642994 0.043311823 0.22012898 -0.17406796 -0.20795141 -0.0003526859 -0.21971963 -0.028871857 0.033430953 0.3162239 0.22255453 -0.12580688 0.10653032 0.0704887 0.06740884 -0.1529267 0.3072419 -0.010793122 0.21301503 -0.15703118 -0.2820408 0.011131795 -0.100724384 -0.016323704 0.2168342 0.08558102 0.27000135 -0.01864184 0.030813348 -0.1455529 0.10483434 0.114837706 -0.16376676 0.11870294 -0.0086970385 0.25253895 0.3348962 0.16174929 -0.34379828 0.26401553 -0.15910603 0.112310536 0.1542048 -0.18695524 0.19872883 0.23120207 -0.11823239 -0.063696004 -0.06734481 0.19670632 -0.082111955 -0.021676552 0.042183448 0.014436004 -0.044563882 -0.035216186 -0.28549877 -0.041587885 -0.21666932 -0.05314732 -0.3081621 -0.042244546 0.059953593 0.22579241 -0.031829614 0.085513756 0.18688159 -0.042920172 0.22894904 -0.13859968 -0.004038722; -0.030131953 0.053396232 0.14041811 -0.11762486 -0.10137773 -0.026885215 -0.034481958 -0.11324168 0.16082759 0.24305712 0.296967 0.08507783 -0.074913 -0.20512457 0.0744593 0.31035396 0.18549249 -0.27872828 0.3071774 0.093761876 0.022047285 -0.06756497 0.27396542 -0.037310917 -0.011816172 0.16673969 -0.1130228 0.13342649 -0.083383694 -0.12042421 0.14564662 -0.25161168 0.014917699 -0.04660513 -0.16079855 -0.019903565 -0.1461059 0.26139164 -0.21036334 0.055038165 0.23828213 0.035064276 0.24355754 -0.13406341 0.070240684 0.10983899 0.24711795 -0.24908657 0.32001764 -0.25308654 0.2966624 0.14657918 -0.047245238 0.02108699 0.13485925 -0.011468751 -0.18708703 0.012536616 0.026115 0.10269554 -0.027981184 -0.08333144 -0.043913584 0.10393532 0.036174085 -0.26802188 -0.22679234 -0.25847435 0.12800105 -0.25655663 0.08212809 -0.15451127 -0.071924955 0.24424255 -0.041065246 0.1368454 0.03281479 0.23549768 -0.20024377 -0.309046; -0.1290626 -0.03290258 0.24224806 -0.060438707 -0.01283388 -0.26620683 0.0043421313 0.29689288 -0.11613978 0.24583437 0.17328067 -0.08510826 -0.13945891 -0.19897217 -0.23213945 0.048038475 0.21020363 -0.32051504 0.12264524 0.116522945 0.14189708 -0.121242814 -0.016874356 0.15669791 -0.015010583 -0.09743459 -0.16030177 0.2412983 -0.060596224 -0.112233795 0.071947716 -0.13323392 0.4131109 0.28405905 -0.12919164 0.16266254 -0.29332274 0.4435685 -0.08072362 -0.29844576 -0.05713434 -0.24412437 -0.13476 -0.10988842 0.327375 0.067798726 0.22055016 0.19168761 0.06762265 -0.016732173 -0.09578846 0.250383 -0.21535219 0.10366231 0.053056296 -0.13358276 -0.16460763 -0.2013756 -0.13503125 0.27193382 0.07421826 -0.3753753 -0.22337432 -0.019369882 -0.34431857 -0.07450816 -0.09048854 0.036946766 -0.12574048 0.13368668 0.06219756 -0.27564964 -0.0938026 0.02011373 -0.12205592 0.061848283 -0.039481632 -0.079524465 -0.29041114 -0.17428094; -0.2029498 0.107230864 0.3281532 -0.041363332 -0.26249856 0.0689258 0.33184823 -0.11222868 0.29899415 -0.02435108 -0.020941826 -0.045964487 -0.25239733 -0.08035847 -0.15795979 -0.2404551 0.29400596 -0.074816056 0.23358984 0.29875237 0.22033079 -0.024370171 0.17269877 -0.033417013 -0.29933912 0.1129969 0.029084982 -0.03714472 -0.10061737 -0.081281856 0.3104757 -0.14563501 -0.031153873 0.22130811 -0.017608084 -0.010494872 0.023256218 0.18857694 -0.3088484 -0.0077021685 -0.05734295 -0.09947978 0.019267887 0.0003629603 -0.050301827 0.28820127 0.16791084 -0.319945 0.18967944 0.034157414 0.028974611 0.03193306 0.034384593 0.27927667 -0.009893869 -0.19553664 -0.20288734 -0.33443323 -0.22195007 -0.0059033777 0.26151618 -0.2989396 -0.15639947 -0.0037850407 -0.07954304 -0.2659201 -0.089559734 -0.15226054 0.1894285 -0.32030478 0.051424135 -0.15879983 0.15463585 0.26175594 -0.029841585 0.29968646 -0.2402788 -0.025173848 -0.14257792 -0.32838348; 0.28100345 -0.033230357 0.035488196 0.12507066 -0.097316325 0.13279948 0.009723365 -0.11447675 -0.11546982 -0.28100172 -0.22781323 0.055269014 0.32595718 0.15465671 0.11900687 -0.18060067 -0.28780776 0.13911875 0.1858502 -0.16837755 -0.004460777 -0.044059306 -0.11683325 -0.36743775 0.03202606 -0.06962248 0.37390077 -0.30843705 0.28932774 -0.09141125 0.18453673 0.027940977 -0.11004233 -0.20920134 0.29872093 -0.026118744 0.032413803 -0.3343901 0.18298018 0.21761686 -0.15295112 0.25464797 -0.11116662 0.07348599 -0.38041493 0.11550566 -0.16226019 0.06263536 -0.007793118 0.16353166 0.076379746 0.029157607 0.025328796 -0.263258 -0.18217643 -0.022097763 0.21798588 0.23513402 0.14529659 -0.39984155 -0.19278874 0.19069114 0.1850641 -0.18893772 0.10475582 0.15157185 0.3560453 0.16891178 -0.09644468 -0.09621381 -0.13520278 0.1958362 -0.13092622 -0.12938273 0.3381187 -0.10322625 0.0038213355 -0.06767631 0.016260881 0.18925813; 0.30718294 -0.23440734 0.06481469 -0.004752419 0.3302562 -0.21718836 -0.24115619 0.030482972 0.016280543 -0.09308015 -0.1781298 0.12114678 0.062983975 -0.020123973 -0.08855541 0.22238079 -0.22853677 0.08812881 -0.08393787 -0.2115586 -0.05229559 -0.07818384 0.03522027 0.12972914 0.06954311 -0.31986165 -0.04345854 0.08453647 0.091338255 0.15980057 -0.06497647 -0.2595221 -0.16711123 -0.05358918 0.07650589 -0.0061289137 0.1682632 -0.056473438 -0.062404048 -0.03980012 0.08948858 0.032060247 -0.10003728 0.06485674 -0.104725145 0.010421905 -0.046946276 -0.06615524 -0.11789604 0.20119852 -0.089654 -0.08912235 0.11021622 0.1238951 0.056196444 -0.015995154 0.11640008 0.11827392 0.020731708 0.16653678 -0.0085761165 0.17767027 -0.1980766 0.021822559 0.061406724 0.18383366 0.06669387 0.23276183 -0.09583532 0.016771622 0.3552918 -0.13441883 -0.18000655 0.038490996 -0.13736838 0.09740845 -0.13593328 -0.22453333 0.062808946 0.16033588; -0.021335335 0.019256115 -0.042090394 0.072071105 0.079740174 -0.22088507 0.034642525 -0.061877243 -0.34126678 0.0012550481 -0.11065434 -0.07467561 0.12325531 -0.010348295 0.08460561 0.13491102 0.10919125 0.15026312 -0.0149281835 -0.13401794 0.109836385 0.06550249 0.04616162 -0.23774576 0.18101831 0.0049176672 0.016887132 -0.028297786 0.17059554 0.13438399 -0.25029597 0.031744447 0.10896522 0.057553094 0.10771811 -0.07596892 -0.098410346 -0.0653117 0.10673814 0.07562039 -0.07833722 -0.14261653 -0.13445902 0.056031868 -0.01283621 -0.24366117 0.003306136 0.1090297 -0.27653256 0.047325395 -0.17812994 -0.25374895 0.094752096 -0.09499308 -0.018070431 -0.0053309635 0.11931269 -0.116360545 -0.10663089 0.22577576 -0.09269581 0.1864055 -0.17020383 -0.23793276 -0.053714808 0.2821965 -0.004840539 0.0857569 -0.09275264 0.18959567 0.022090692 0.20490974 -0.023580572 -0.11772431 -0.11963751 0.12151551 -0.0025087085 -0.0665179 0.28381675 0.046438288; 0.055613235 0.06198385 0.16012572 -0.21404192 -0.016294938 0.08634883 0.25027373 -0.0671795 0.12083302 0.18404956 0.03412596 0.08342541 0.037417192 -0.38433808 -0.0032337694 -0.1332807 -0.010612142 -0.35492286 -0.12457835 -0.020821018 0.104593135 -0.24022813 0.18498531 0.28957367 0.010413202 0.26514134 -0.14582948 0.07905166 -0.2659199 -0.037493087 -0.18384553 -0.07936706 0.08135873 0.268224 -0.11768431 -0.027254509 -0.2886862 0.30854422 -0.2002917 -0.3304892 0.3226411 -0.025456326 -0.10877496 -0.11705089 0.33938047 0.1474732 0.1737879 0.16728412 -0.0904848 -0.012100644 0.14150825 0.30574584 0.043562848 0.120994106 -0.009931953 -0.017756118 -0.010348492 -0.21880116 -0.20882815 0.17394578 0.38189027 -0.17911814 0.00049492053 0.29665825 -0.06726888 -0.1585042 -0.3192908 -0.29630923 0.04275041 -0.092340685 -0.08699201 -0.15611118 0.04785565 0.17095664 -0.1572075 0.23770037 -0.10982592 0.1988413 -0.15199535 -0.27089933; 0.11054047 -0.0010077105 -0.25490913 -0.020142319 0.062115625 0.0649331 -0.020030404 0.05875001 -0.11324634 -0.26329646 -0.040382467 -0.046845317 -0.014463118 -0.042433973 0.07705555 0.12308298 -0.05287147 0.12452943 -0.3402525 -0.019877268 -0.17000484 0.022272857 -0.26938272 -0.33522266 0.07451806 -0.02141319 0.13570061 -0.25518858 0.2917728 0.337828 -0.030730195 -0.11689966 -0.07978361 -0.20526806 0.13177033 -0.27762765 -0.051261712 -0.1857761 0.059911344 -0.014739996 0.06059323 0.076288685 -0.12399429 0.026610931 -0.23198846 0.047012866 -0.015970483 0.17139652 -0.25491 -0.08219133 -0.030673333 -0.18442589 0.019583 -0.16525675 -0.100353174 0.03402945 0.19151995 0.21251437 -0.13834146 0.034949917 -0.116963975 0.12672684 -0.12131834 -0.22588396 0.21934089 0.17675497 0.14879243 0.20064878 0.012249848 0.23022044 0.19800302 0.21749088 -0.3042606 -0.14172943 -0.010255447 -0.2196632 0.13090616 -0.030810433 0.28064275 0.35323033; 0.16933292 -0.15822819 -0.27182296 0.34139973 0.196769 -0.020169482 -0.05814118 -0.02240232 -0.21053995 -0.30909568 0.023019731 0.20191392 0.15257987 0.36931363 0.1851534 -0.21331166 -0.17635076 0.22103445 -0.109762274 -0.08874599 0.038015287 0.2575255 -0.2829267 -0.35262147 -0.21309742 -0.09267845 0.33367985 -0.31711185 0.055582546 0.052499197 0.10072407 0.2182109 -0.08426848 -0.18690443 0.16725597 -0.22843704 0.2261455 -0.11899142 0.25557235 0.23695612 -0.2647797 0.31632265 0.028501032 0.1502116 -0.36817563 0.0037917236 -0.2801196 0.101691805 0.097204275 0.24936129 -0.1657037 -0.2010375 -0.044700015 -0.312468 -0.25799954 0.0047898004 -0.07223449 0.097199485 0.28455842 -0.06918524 -0.3628108 0.08547933 -0.046372857 -0.24095078 0.16825487 -0.01972853 0.20789875 0.19794501 -0.00809674 0.0024174459 -0.1264471 0.12330857 0.09153013 -0.20857105 0.27837712 -0.22784111 0.23656423 0.0063568065 0.31151026 0.3456168; 0.27331263 -0.22705667 -0.10342557 0.073469095 0.15237235 -0.06787145 -0.23673397 -0.029346742 -0.07972954 -0.036049623 -0.051140875 0.06905185 0.07074741 0.041571893 -0.10658948 0.048176855 -0.24517074 0.020715328 -0.23796582 -0.052459955 0.059598804 0.23935133 -0.17498127 0.015748879 0.08806147 -0.27343205 -0.0020410921 0.012037338 -0.018589936 0.078299895 -0.0020095431 -0.15686014 -0.26123297 -0.16903397 0.06889236 -0.2713636 0.24272333 -0.09795295 0.04743595 0.25143787 -0.14760934 0.15394227 0.039749824 0.12740299 -0.2108779 -0.27065855 -0.19553182 0.21107861 -0.16464728 0.14188708 -0.29587275 -0.30594894 0.031073052 -0.05500171 -0.19274695 0.14794171 0.053648725 0.10063824 -0.21079317 -0.11813992 -0.16420916 0.18182918 0.08816208 -0.14572287 0.23899004 0.004817088 -0.07152672 0.16256648 0.0070596985 -0.0009392091 0.118549556 -0.0026806148 -0.2013885 -0.1340539 0.18709816 0.026254302 0.07834743 -0.17818326 0.25887457 0.114827044; -0.02533475 -0.27941754 0.015773024 -0.067306675 0.12310958 -0.112138025 -0.055760894 -0.012252321 -0.28051266 -0.277429 -0.05680162 -0.14898917 0.1645564 0.0029411872 -0.22878042 0.21994361 0.011804613 -0.13311537 -0.3171241 -0.014224995 -0.023828818 -0.024641607 -0.1880439 -0.24958645 0.17642681 -0.16775167 -0.038234122 -0.22080399 -0.020882841 0.09691186 -0.24224037 -0.018987762 0.01474459 -0.042427395 0.020039447 -0.14642239 0.20376906 0.018449882 0.091477275 -0.09550007 0.0034108898 0.20644763 -0.2831964 0.067854434 -0.15569296 0.0019832933 0.047295406 0.08964051 -0.11670149 0.22454396 -0.19204591 -0.11584438 -0.030260576 0.23425217 -0.29360244 0.17544933 -0.08559933 -0.04271422 -0.044547565 -0.0036429064 -0.17213373 0.09453301 -0.12081066 -0.09416252 -0.1414095 0.23907039 0.07892864 0.102819614 -0.16495155 0.2854258 0.048178773 -0.046974555 -0.16796263 -0.061205752 0.16894145 -0.19673233 0.04698129 -0.2412842 -0.0069626654 0.2648511; -0.12857145 0.09336411 -0.11690495 -0.09208089 -0.3017301 0.08780599 0.17768262 -0.20777126 0.15701912 -0.18119033 -0.13262208 0.08353705 -0.14318496 0.14201641 0.08553935 -0.048080098 -0.09012106 -0.08914509 0.04305295 0.16182636 0.027975153 -0.00866792 -0.19661373 -0.2426338 -0.36041674 0.20535502 0.13844 -0.16574124 -0.14799044 -0.08176033 0.26169267 0.13732547 0.06955104 -0.16500689 0.22936867 -0.10262715 0.18006526 -0.11599758 -0.037708186 0.1849317 -0.046877768 -0.022896562 0.16223931 -0.062687725 0.01598805 0.11068095 -0.042140048 -0.04444984 0.28100663 0.04952546 -0.04231846 -0.16189162 -0.20468947 -0.09712388 -0.047312573 -0.11190876 -0.24524583 0.23359147 0.11492688 -0.20741582 -0.1675212 -0.04955251 0.008565147 0.112441346 0.13630518 0.0014744417 -0.076326445 0.04510202 0.23454428 -0.2846376 -0.31259766 0.012967333 0.27222103 0.07271511 0.21845846 0.099130034 0.100933224 0.21987644 -0.13216238 -0.047596425; -0.29024538 0.11986409 -0.05006376 -0.17086081 -0.13422999 0.11247704 0.32183447 -0.07506159 0.060054433 0.23361334 -0.0916632 -0.041522253 -0.1452821 -0.085300475 -0.14041266 0.07547826 0.007929355 -0.034162585 0.15119307 0.08944607 0.17431533 0.07360091 0.26974916 0.043926395 -0.1560384 0.16884509 -0.2771871 0.17057793 -0.09207397 -0.27021405 0.08751791 0.034166425 -0.014065184 -0.005436077 0.016696239 -0.00078454963 0.075809024 0.013281553 -0.0323751 -0.18924996 -0.05357559 0.00980897 0.014653545 -0.15961504 0.2975917 -0.0071442192 0.031113716 -0.038825925 0.2332649 -0.18648456 0.21926033 0.12466307 -0.18244167 0.2096363 0.13790783 -0.29609913 -0.2811236 -0.02911407 -0.03153926 0.17843112 0.062666856 0.008510175 -0.003628278 0.055294026 -0.073873505 -0.22926809 -0.25288364 -0.14600633 0.28225368 -0.053187426 -0.29956412 0.042330638 0.20200485 -0.058048572 -0.2714954 -0.07470335 0.070586056 0.32777208 -0.016876662 -0.10428645; -0.112853564 0.10930016 0.03950841 -0.17800419 -0.08039562 -0.04398746 0.22465311 -0.087517835 0.17592122 0.24802184 0.19479303 -0.18132786 0.041613247 -0.10698776 -0.23620038 -0.20653124 0.023631545 -0.3073831 0.266753 -0.00014632438 0.06960776 -0.26796153 0.004141745 0.17558323 -0.117684156 -0.010681389 -0.16407944 0.27931836 -0.24391757 -0.13397312 0.25938934 -0.086884424 0.047994 0.07956691 -0.23164059 -0.0150986165 -0.007740338 0.24259159 0.0601451 -0.014997805 0.025218416 -0.33937573 -0.04060233 -0.04269674 0.22438926 0.18502748 0.31949848 0.04248269 0.16759285 -0.095267884 -0.036599893 -0.031007927 -0.022831943 -0.017494578 0.30525824 -0.00909732 0.00057177804 -0.1966347 -0.22790745 0.33134285 0.27171996 -0.32188928 -0.14779876 -0.048642248 -0.07424216 0.04753845 0.031211212 -0.22517271 0.23733242 -0.058609318 -0.056002595 0.02874129 0.110961586 0.193664 -0.22373083 0.18993133 -0.24020346 0.21809424 -0.26072043 -0.1320712; -0.08539591 -0.017591912 0.023898996 -0.26371473 -0.061319992 -0.25132883 0.11239295 0.20875283 0.078552954 0.27157006 0.23355573 -0.18308517 -0.22516795 -0.3412955 0.044624027 0.092276506 0.14372535 -0.25239587 0.22172828 -0.09903387 0.18800212 -0.16919889 0.2809108 0.30303404 0.24259943 0.2081769 -0.18663822 0.3277345 -0.19206335 -0.14714795 0.05478471 -0.104412556 0.08489529 -0.05007093 -0.2504777 0.06900903 -0.0030627658 0.11497676 -0.20739749 -0.35149422 0.3033285 -0.28065524 0.15599483 -0.37176043 0.075932086 -0.054471415 0.15250875 -0.15645644 0.1179657 -0.27852768 0.21061464 0.020823926 -0.32668254 0.19019702 0.21275045 -0.12259957 0.008751806 0.06405422 -0.16072193 0.32470652 0.20059614 -0.09107432 -0.116609946 0.2030927 -0.13430937 0.0355156 -0.2848054 -0.28572705 0.14790833 -0.112464204 0.086198874 0.0604074 0.1281515 0.12907499 -0.34780788 0.11896902 -0.022761662 0.1894397 -0.013015771 -0.117367685; -0.2885607 0.22240736 0.16183479 -0.082296945 -0.27885598 0.22349249 0.13774747 0.07557411 0.06716174 0.11334241 0.21053427 0.24581718 0.036176413 -0.027873075 0.017835272 0.011953296 0.15820949 -0.11604531 -0.0174719 0.1650796 0.1912996 0.074483626 0.20008893 -0.091860875 -0.15474363 0.120906256 -0.09804241 -0.07293958 -0.09233156 -0.09049518 -0.034540977 -0.046975803 0.29689497 0.18395767 -0.008095928 0.33588365 0.084697574 0.15804069 0.12523735 0.12595846 0.08716817 -0.07062835 0.33461985 0.0925121 0.07496001 -0.015463904 0.035583027 -0.18795827 0.23991287 0.07076428 0.15200809 0.0057458957 0.031078143 0.04998441 0.116442606 -0.2683957 -0.27575925 -0.23070048 0.14830722 0.11365313 -0.02526145 -0.15171924 0.018688845 -0.011156705 -0.016002439 -0.019209627 -0.24665307 0.04357387 0.25242287 -0.15338333 0.026701204 -0.014955004 0.052020352 0.25435555 -0.11568076 0.17148036 0.16883183 0.0542231 -0.22939293 0.049494654; 0.06228678 0.049821932 -0.124097876 0.032270595 0.042485077 0.3053666 0.14573555 -0.005220107 0.13066344 0.18447302 0.08894822 0.06983023 0.09864127 0.2266515 0.028887859 -0.27001247 -0.033771556 0.11558675 0.23788589 0.14759378 -0.14428574 0.09629521 -0.087496 0.12955394 0.017944166 0.30657297 -0.15659538 0.1796271 0.072883755 -0.18053468 0.09558433 -0.0966787 0.12801674 -0.1132814 -0.1142374 0.15597597 0.15753435 0.12977573 0.05481873 -0.02651016 -0.06314888 0.075065054 0.14597712 0.06691522 -0.10462152 0.2344857 0.04331451 -0.008303815 0.14063457 -0.020225063 0.18514879 -0.041338146 -0.31052193 -0.20305479 0.07531982 -0.07235464 0.04347333 -0.08374807 -0.04655657 -0.0079751145 0.093818255 -0.14932925 -0.06886107 0.0019846582 -0.029547196 -0.1530588 0.07178754 0.0775177 0.21348114 0.016598796 -0.13620026 0.07487665 0.025452297 0.20088865 -0.12268382 0.013102643 0.21949752 0.053931177 0.031851515 -0.09655081; -0.022542875 -0.01110323 -0.1373312 -0.062089223 0.0067077107 -0.29932404 -0.19226214 -0.00587634 0.015940968 0.052240245 -0.12668636 -0.17575839 0.15320623 -0.16610198 -0.15345488 -0.00051818514 -0.19420548 -0.17865217 0.0016248109 -0.062335506 0.10982669 0.2049832 -0.16830646 0.061944425 0.07969928 -0.16221349 0.13697633 0.067229785 0.2601284 0.18781427 0.0014144371 0.0012761493 -0.15129645 -0.19774616 -0.0438022 -0.33683893 -0.054404166 0.036500257 0.19755226 -0.048695076 -0.07644391 -0.0004928301 -0.04641982 -0.13954192 -0.1236721 -0.15250432 -0.0068137925 0.14762117 -0.03227735 -0.00927327 -0.18999007 -0.060675073 0.31961673 0.18639377 -0.0702089 0.17194092 0.22763214 0.09567306 -0.08914832 0.0994077 -0.025964534 0.29655796 -0.12078577 -0.28641903 -0.09480185 0.114780046 0.18933257 0.003763995 -0.10171992 0.21687439 0.27230987 0.13294671 -0.23696923 -0.06466923 -0.06626889 -0.09591014 0.06108668 0.038455743 0.09117919 -0.024019571; -0.28149933 0.038795862 0.20323166 -0.2206401 -0.12721574 0.14887585 0.11363254 -0.15884107 0.020785 0.1281904 0.096545614 0.27965677 -0.07831376 -0.06718568 0.006773987 -0.047360353 0.22780022 -0.0695022 0.14398977 0.029194014 0.005322825 -0.033588853 0.11844934 0.081585556 -0.035969265 0.25342366 -0.120302476 0.113659434 -0.20035976 -0.16692036 0.14933883 -0.010186221 0.27321917 0.14645371 -0.22735986 0.068725035 -0.17407541 -0.038337544 0.13445519 0.09240689 0.011626313 -0.19921829 0.24777985 0.08915889 0.21284053 0.06957163 0.1990209 0.00877178 0.18121776 -0.18001252 0.08497372 0.17505117 -0.13339765 0.0038540626 0.12877011 0.023032181 -0.13953489 0.011639021 0.15411043 0.08387625 0.070156574 0.0042488794 -0.033355955 0.3174834 -0.24313208 -0.11932635 -0.000469139 -0.23409677 0.1999184 -0.118927605 0.031014003 -0.10879884 0.038441576 0.30978093 -0.16652854 0.029278249 0.01715099 0.045240656 -0.13674708 -0.048491653; -0.201999 0.12228256 0.12169199 -0.18399374 -0.17776412 0.32292792 0.31783262 0.1091987 -0.0002948238 0.19534923 0.25373325 0.161301 -0.27259895 0.030514253 0.1729168 -0.07917576 0.13211478 -0.0011802296 0.032200314 0.2543682 0.18771167 0.051720697 0.23542573 -0.08027554 -0.25387204 -0.02523275 -0.09809793 0.028491223 -0.15491869 -0.06757695 0.0037390802 0.14377788 0.2829499 -0.100086324 -0.12724382 0.075702935 -0.08187973 0.051413804 -0.13346124 -0.11525295 0.022080835 -0.2344252 0.03646816 -0.10489926 0.16820376 -0.0057843095 0.0943923 -0.19434536 0.2905024 -0.08372048 0.2663489 -0.05792659 -0.06321755 -0.12004568 0.282275 -0.010991795 -0.15035054 -0.24333836 0.10196655 -0.10132657 0.16542046 -0.004625917 -0.15357204 0.14577062 -0.11779237 -0.29311913 -0.2805147 -0.08587775 0.04493683 -0.22221397 -0.2671264 0.011241043 -0.0044060103 0.11093073 0.05038645 0.2692094 -0.00962495 -0.03064089 -0.08976671 0.021023728; -0.2881457 0.19380407 0.11179741 -0.02970951 -0.17053327 0.05157511 0.0076164035 -0.0749169 -0.034131717 0.18650897 0.03020034 -0.08735717 -0.12710753 -0.014157293 0.00167643 0.13165236 -0.037442736 -0.05856038 0.24635057 -0.019411068 0.023421567 -0.22205369 0.0160832 0.33654588 0.07197728 0.009288414 -0.27880812 0.141209 -0.025429776 -0.28363067 0.08308346 -0.32848838 0.3474613 0.1782334 -0.2898816 0.08957751 0.019425608 0.17955658 -0.25245148 -0.30992603 0.20697996 -0.20786107 0.2538629 -0.0032988107 0.039088998 0.20613612 0.16878389 -0.11375238 0.015222865 -0.2583317 0.2295741 0.0020701205 -0.2612825 0.197045 -0.018269235 -0.0980916 -0.08386835 -0.109573536 -0.24004191 -0.0031889672 0.26473242 -0.081648484 -0.11423199 0.2710999 -0.11645648 -0.06767332 -0.20086184 0.026110642 0.034421936 0.05565676 -0.027628092 -0.1861098 0.010311413 0.046774145 -0.0108582 0.15558462 -0.10886967 0.053007357 -0.1076411 -0.19438988; 0.024208345 0.02941601 -0.29024988 0.1356676 0.054898743 0.016381117 -0.08210096 -0.018677985 -0.17792828 -0.09954001 -0.30643487 0.06654948 0.16928048 0.055610362 -0.019675981 -0.33984038 -0.21034184 0.22816178 -0.14189829 -0.15550496 -0.18357293 0.18152684 -0.09633814 -0.135493 -0.3529573 -0.2472004 0.2753424 -0.065606095 0.23175892 -0.0037382492 -0.12207696 0.22933389 -0.33410478 0.030254675 0.10895799 -0.061736155 0.23313089 -0.029199833 0.06281325 0.22908229 -0.17980875 0.33913895 -0.03528918 0.045968533 -0.15284337 -0.1452688 -0.21554455 -0.18016335 -0.18128552 0.026760805 0.07846435 -0.27946907 0.15603825 -0.057397265 -0.03281928 0.19279043 0.13880524 0.1082858 0.05371481 -0.24955098 -0.2665173 0.15764496 0.034290757 0.09681271 0.18302414 0.21889894 0.24257453 0.2086982 -0.24046716 0.13023771 0.10817169 0.019571094 -0.14367972 -0.20581442 0.23424068 -0.052295357 0.28083435 -0.27504727 0.07282096 0.028142747; -0.2055116 0.059214815 0.11948504 -0.06875882 -0.3362236 0.02942389 0.24794775 0.06508705 0.22434038 0.08728046 -0.05818173 -0.047557324 -0.25074404 -0.2684736 0.11544099 -0.4826868 0.04982379 -0.09900466 0.19188222 0.048529454 0.045608692 -0.21380815 -0.049321584 -0.017111622 -0.069999844 0.14192034 -0.010230942 0.0053046886 -0.10884952 -0.024387222 0.27385202 -0.16594405 0.107035816 0.0517575 -0.077344365 0.1821509 -0.17206186 0.02900853 -0.278969 -0.070894815 0.23498222 -0.1881843 0.06342492 -0.19190674 -0.06397058 0.28347707 0.0050928677 0.011264581 0.12652689 -0.1833143 0.29275092 -0.021962147 -0.3184578 -0.0798804 0.14770968 0.02324823 0.0012327108 -0.11397004 -0.08106489 0.105033554 0.23074417 -0.16177997 0.08412929 0.26824084 -0.22121416 -0.021874176 -0.1566992 0.011933632 0.17539926 -0.23831587 0.004752146 0.06352606 0.18294963 -0.04841705 0.0020266213 0.08473106 -0.12159039 0.16508132 0.048931368 -0.24515668; 0.036998793 -0.13422437 0.03200058 0.107878804 0.27217102 -0.02770946 0.038863096 0.13013056 -0.28080252 -0.19271196 -0.15250488 -0.180386 0.05181346 -0.14318588 0.09592763 0.16553897 -0.13751756 0.11844996 -0.12643656 -0.025770593 0.06965929 0.1206755 -0.016767519 -0.020126631 0.29838687 -0.15953489 0.07872257 -0.057127483 0.267171 0.2921512 -0.0031682428 -0.054950636 -0.062345855 -0.06652449 -0.13562474 -0.23877017 0.28218663 -0.0655458 0.0223888 -0.02813122 -0.12029467 0.038250867 -0.1332307 0.12455067 0.05497099 -0.21284431 0.0057856017 0.029348206 -0.03639425 -0.02898325 0.0053186817 -0.13784721 -0.00021502604 -0.0531502 -0.11216596 0.2535894 0.17298871 0.08900881 0.099276185 0.14451683 0.079423845 0.2110914 0.10859666 -0.25322902 0.03968131 0.2736957 -0.0047804364 0.1745294 -0.15924394 0.23319305 0.14300121 0.13870382 -0.1186428 -0.24606802 -0.034457408 -0.2545623 0.10389387 -0.2014263 0.25641295 0.1612195; 0.09638248 -0.22711699 -0.03770056 0.013787311 0.07892013 -0.094500795 -0.18757051 -0.034939136 -0.070780024 -0.1580038 -0.115387745 -0.08963374 0.10958489 -0.032317936 -0.122135125 0.24417783 -0.035341546 -0.147532 -0.20277232 -0.19942883 -0.09026089 0.07924346 -0.19163986 -0.11878453 0.004112954 -0.14382873 -0.009288484 -0.038649064 0.08415193 0.19629698 -0.34400356 0.087179065 -0.094703 -0.107574314 -0.066876255 -0.20765114 -0.0055135884 -0.20731235 0.16948912 0.049505524 -0.0809477 -0.0117235575 -0.26054323 0.066805616 0.023023138 -0.00530818 -0.043111354 0.02041071 -0.24798366 -0.0111231385 0.009145621 -0.10271612 0.22148076 -0.047122873 -0.061736945 0.08646664 0.08514669 -0.073793024 -0.16569558 -0.198977 -0.09698496 0.16562714 -0.08119978 -0.28795844 0.24455836 0.038965777 -0.031251296 0.0037404022 -0.04757763 0.17104398 0.1210451 0.051818635 -0.28222513 -0.22142859 -0.056580298 0.060326446 -0.09177641 -0.03590731 -0.029661328 0.2694506; 0.28365934 -0.04566966 -0.16881616 0.123462774 0.09305 -0.22664407 -0.24538405 0.20464692 -0.026989982 0.17709255 0.2182911 -0.12417197 0.18085064 -0.10161651 -0.1325838 0.0772825 0.00639903 -0.1665689 -0.10696896 -0.026282014 -0.032736067 -0.07212399 0.12431175 -0.05346547 -0.023466792 -0.28059176 0.034382712 0.014152454 0.14980769 0.05830491 -0.27190432 0.009653092 0.17421229 0.06968502 0.067999706 -0.19933967 -0.018212955 -0.20902233 0.1187128 0.1225178 -0.094212376 -0.032233737 -0.19522877 0.020614292 0.18903054 -0.16139653 0.09433168 0.12812635 -0.035631243 0.029131992 -0.04519771 -0.11066549 0.17218342 0.08783691 -0.15283498 0.14586383 -0.08554124 -0.008326389 0.05912507 0.19633402 0.060898587 0.009472844 -0.09774087 -0.2264201 0.016543958 0.14095968 -0.08549228 -0.10348874 -0.041835718 0.11309599 -0.050930936 -0.022790512 -0.01087089 -0.2665959 0.08619247 -0.12573569 -0.13216694 -0.27605313 0.1050349 0.15485576; -0.029097915 -0.2952573 -0.28496695 0.061778888 -0.023338493 -0.27593845 -0.113852955 0.021102028 -0.24444342 -0.12269877 0.060417473 0.018493332 0.2934638 -0.08246987 -0.13455167 -0.0741327 -0.20633972 -0.07415033 -0.27088764 -0.20372191 -0.047508314 0.1894451 -0.095785476 0.033291806 0.0458982 -0.31482917 0.037812345 -0.22181012 0.017272811 0.29073575 -0.17273404 0.15073189 -0.2817726 0.118013114 -0.012853764 0.033377353 -0.08635029 -0.15239948 0.0033606614 0.08684705 0.10719726 0.0848251 -0.12025934 0.19123532 0.033910617 -0.100591294 -0.113848135 0.013842357 0.037896343 0.21548383 -0.2749718 -0.13373706 0.3159481 0.061961744 -0.099658474 0.091581255 0.27388814 0.21600555 -0.0039085154 0.0749745 -0.042914964 0.11300344 0.03497983 -0.31969568 0.011336154 0.13995683 0.093740955 0.28640062 0.08687446 -0.053649765 0.30479857 -0.013095294 -0.31793383 -0.1763389 0.080331825 -0.14788955 0.027502138 -0.26168862 0.2758985 0.042729285], bias = Float32[-0.07979798; -0.053887736; 0.03923683; -0.021176286; -0.027510077; 0.053404517; 0.0057087606; -0.0015105834; -0.052914582; -0.028157162; 0.037350472; -0.042782404; -0.026713092; -0.044450305; 0.018365018; 0.052158058; 0.006992461; 0.049514182; 0.055971656; -0.052901447; 0.038692787; -0.035480045; 0.035752203; 0.0063355644; -0.06857357; 0.052899066; -0.0694431; 0.019140903; 0.05108501; 0.061331786; -0.094744764; -0.020183556; 0.08000746; 0.055342726; 0.04879285; -0.07866074; -0.014690223; -0.0102303885; 0.03701071; 0.05055103; 0.05072233; 0.022581384; -0.04913802; -0.039277337; 0.044977006; -0.23984624; 0.05228278; 0.04991659; 0.05234421; -0.020274766; -0.023137236; 0.050314564; -0.037044447; -0.049104914; -0.04082803; -0.06150115; 0.03890562; 0.06320512; 0.05307957; -0.015433883; -0.06559534; 4.82984f-5; 0.018331403; 0.033703193; -0.03349886; -0.030776208; -0.038892154; -0.0339789; -0.041188747; -0.044416834; 0.057721544; -0.037608355; -0.044453174; -0.05087957; 0.012602862; -0.066933095; 0.06963772; 0.052535884; 0.06258303; 0.02337812]), layer_3 = (weight = Float32[-0.11843466 0.060342666 0.07411992 -0.004693234 -0.053191476 0.08611754 0.21408385 -0.22675897 0.0020225933 -0.053081892 -0.06541322 -0.025490731 0.14897996 0.10074279 -0.04948093 0.09152146 -0.028996034 -0.08744151 -0.08249433 -0.05592985 -0.09305496 -0.08635692 0.08000045 -0.12230062 0.1155688 0.013381793 0.09913415 -0.2429029 0.039542902 0.15991257 -0.16981791 -0.16534252 0.046755984 -0.04619362 0.016172083 0.041305084 0.19668019 0.056304675 0.1678263 -0.01996038 -0.11851451 -0.15785456 -0.03274484 0.030059736 -0.009069605 -0.14830668 -0.15621914 -0.15449966 -0.10729438 -0.22994465 0.16675913 -0.0003452572 0.003473957 0.10398907 0.15260121 -0.13699104 -0.061587747 0.08701657 0.11699575 -0.078403495 0.07118849 0.008263703 -0.1418298 -0.091645256 -0.11017523 0.17237383 -0.10203199 0.04614932 0.14422959 0.06389608 0.043427892 0.18162873 0.0040077562 -0.08154534 -0.0076869056 -0.098317005 0.11557372 0.008336721 0.008441393 -0.19180085; 0.10420133 0.0529325 0.112449065 -0.0031898448 -0.23508814 -0.026909072 0.173957 -0.315816 0.07204785 -0.06415942 0.40389025 -0.20655237 0.2528334 0.2531368 -0.044641417 -0.021647977 -0.09797406 -0.12349889 -0.121384025 0.056679428 -0.17130403 -0.3103379 0.22163662 0.08807301 -0.2342196 0.058719542 0.29503447 -0.1940656 -0.006158679 0.10128441 0.06497948 -0.17807229 0.16838749 -0.30738577 -0.030229686 0.18339549 0.36524272 0.07439612 0.023120841 -0.095802225 -0.3116285 -0.15848231 -0.08375955 0.06146601 0.17580205 -0.1291319 -0.16132052 -0.09545135 0.22196214 -0.19292575 0.16915813 0.0033855196 0.26958963 0.08223013 -0.0885018 -0.023522269 0.3111847 0.08440407 -0.005363844 -0.29486272 -0.0068663447 0.20276448 -0.153871 -0.23893505 -0.18267281 0.07897001 -0.1811735 -0.20315339 0.18846644 0.019468144 -0.07621422 0.12921807 0.11185677 -0.103233926 0.014770519 -0.07382284 -0.11845171 -0.029915309 -0.027797934 -0.34251738], bias = Float32[-0.3889578; 0.0025400005]))


loss_neuralode(p_trained_nonoise)


#--------------------Moderate noise--------------------#
#-------------------------------------------------

rng = Random.default_rng()
Random.seed!(99)

#Adding moderate noise to data:

x1=Array(sol)

x1_mean = mean(x1, dims = 2)
noise_magnitude = 7e-2
x1_noise = x1 .+ (noise_magnitude*x1) .* randn(eltype(x1), size(x1))
#Displaying true data vs noisy data

#-------------------------------------Defining the Neural ODE------------------------------------


dudt2 = Lux.Chain(Lux.Dense(2, 80, tanh),Lux.Dense(80, 80, tanh), Lux.Dense(80, 2))
#Setting up the NN parameters randomly using the rng instance
p, st = Lux.setup(rng, dudt2)


prob_neuralode = NeuralODE(dudt2, etaspan, Tsit5(); saveat = etasteps)

function predict_neuralode(p)
    Array(prob_neuralode(I, p, st)[1])
end


#training data
true_data= x1_noise[:,1:end-60]
### Define loss function as the difference between actual ground truth data and Neural ODE prediction
function loss_neuralode(p)
    pred = predict_neuralode(p)
    loss = sum(abs2, true_data .- pred)
    return loss, pred
end

p_trained_moderate = (layer_1 = (weight = Float32[0.07935157 -0.21924509; -0.273308 -1.1245617; 0.09500121 0.44475293; 0.02935812 -0.023323148; -0.04166859 -0.35314476; 0.38528365 0.6210743; -0.18736798 0.7259024; 0.10685388 -0.8924722; -0.5379336 -0.44419327; -0.4373724 -0.2757848; 0.36478832 0.5575491; 0.30163231 1.2519413; -0.30612627 -1.2182641; 0.400033 -0.35118535; 0.43311837 0.65844464; -0.358416 -0.34608614; -0.11323094 -0.27764267; 0.0721443 0.119139664; -0.26553982 -1.0164204; -0.02216973 -0.53219205; 0.53672427 0.7684129; -0.20425071 -1.3169146; -0.527978 -0.5054698; -0.49422947 -0.22768104; -0.33426863 -1.1902822; -0.2636758 -0.17368303; 0.117352545 -0.49150562; -0.40048152 0.76735854; -0.5542262 -0.739574; -0.0187035 -0.56363297; -0.28305238 0.06749907; -0.6043819 -0.10540653; -0.4101286 -1.0692652; 0.31467524 0.9871122; -0.06799528 0.07751108; -0.5820319 -0.12286143; -0.32768917 -0.43092644; -0.43275344 -0.028461043; 0.07360572 0.53521174; 0.03168502 1.2396483; 0.088475674 0.1593816; 0.20215365 0.18894725; -0.4549379 -0.9419576; -0.13100754 -0.2797622; -0.09470803 0.23869933; -0.26790485 -0.12977329; 0.387307 -0.8685889; -0.051719714 0.6457771; -0.50591934 -0.0017826259; -0.20152877 0.30786785; -0.40008104 1.007239; -0.029872004 0.49101815; -0.07773946 -0.017899657; 0.18630305 -0.48822105; -0.06646489 -1.1040348; -0.7141009 -0.6954117; -0.24777474 -0.17337576; 0.29374713 1.1825049; 0.54568875 0.26335743; 0.02997594 0.30694184; 0.19614588 0.038962815; -0.10108959 0.17498273; -0.2084132 0.5687564; -0.39318508 -0.580487; 0.037957266 -0.32401568; 0.5953569 0.2850247; -0.3252331 -0.63900554; -0.37291217 -0.63284445; -0.11976367 0.053025268; 0.5416413 0.77104765; -0.026423462 0.118760556; 0.20860541 1.2926133; 0.19057457 -0.6369097; -0.12284818 0.45633018; -0.37207103 -0.36379948; -0.014515924 -0.18862674; 0.6948389 0.57195574; -0.06594579 -0.20935254; -0.14070867 0.04856727; 0.36758274 0.89992785], bias = Float32[0.25715822; 0.08730745; -0.39010632; 0.31033045; 0.25227335; -0.29821712; 0.3193414; -0.41044062; -0.07077982; 0.2552034; 0.16548744; -0.13593926; -0.03638866; -0.07744417; -0.123140685; 0.13725808; 0.44266805; -0.4100616; 0.28754294; -0.32650545; -0.00016660127; 0.04299185; 0.39510664; 0.040502287; 0.19889358; 0.3927007; 0.37975216; 0.25971416; -0.082625; 0.2073752; -0.39496756; 0.22130403; 0.022936529; -0.13282257; 0.16547188; -0.19926259; -0.12441051; 0.25235847; 0.41059232; -0.06916621; -0.21881066; -0.24610275; -0.06329596; -0.30764592; 0.25272655; -0.4340389; -0.28313485; 0.08268906; 0.32825527; -0.2364427; 0.3463973; 0.29702255; -0.2072201; -0.2732226; -0.111650966; -0.3618056; -0.4523558; 0.24183536; -0.26910067; 0.36624873; -0.31879288; -0.4195971; -0.2236629; 0.16971764; -0.29166302; 0.09604283; -0.048080195; -0.16605337; -0.3022104; 0.32391822; -0.3605036; -0.35727197; 0.16226618; 0.35609725; 0.010739266; -0.23343545; 0.4134773; 0.40176293; 0.20316108; 0.038832325]), layer_2 = (weight = Float32[0.4388501 0.34927335 -0.37074146 0.21699421 0.2927166 0.020961605 0.059948884 0.36818463 0.14739196 0.37695372 -0.19629107 -0.23730433 0.16284366 -0.22811604 -0.10236605 0.18590155 0.24686806 -0.24492809 0.22843948 -0.3282367 0.0034322035 0.21627398 0.09602799 0.08422039 0.07938692 0.29167056 0.30453604 -0.45113784 0.16485855 0.320507 -0.312168 0.026439762 0.27390805 -0.26433575 0.24787062 -0.06829427 0.081146464 -0.22895847 0.27560517 -0.42974895 -0.4074976 -0.28927615 0.04884022 -0.19661158 0.19838285 -0.5710158 -0.023213925 0.24640875 0.06393298 -0.54651225 -0.11005762 0.4361546 -0.3177142 -0.34411275 0.4328854 0.18663107 -0.19334494 -0.36172634 0.04697728 0.15669161 -0.39252302 -0.47358087 -0.5832672 0.06978688 -0.36786258 -0.1806724 0.22785163 0.040624 -0.30890605 -0.036028195 -0.3922983 -0.4238424 0.58911866 0.11693239 0.14830327 -0.45566988 -0.28196123 0.28682753 0.27042836 -0.11389345; 0.45146587 0.11066623 -0.4069646 0.25749436 0.15956196 -0.08067391 0.20212069 0.6635979 0.042609043 0.34323665 -0.30548912 -0.29014665 0.37236184 -0.22267021 -0.12234056 0.3552448 0.50892556 -0.31155795 0.2228133 -0.45630258 -0.04393827 0.39668283 0.19561733 0.16883767 0.09835513 0.23689558 0.37784612 -0.22063543 0.31050944 0.2915538 -0.38279942 0.34706435 0.2660261 -0.29860657 0.3004966 0.074484736 0.37074295 0.12026572 0.2522362 -0.3814226 -0.25733727 -0.14380552 0.027169613 -0.21169125 0.28803426 -0.57034606 -0.070593625 0.4534696 0.23009364 -0.44967064 0.064022765 0.26829675 -0.20898364 -0.25866637 0.2829129 0.22611211 -0.40970606 -0.3925315 -0.39915434 0.41526535 -0.42142537 -0.43013817 -0.5318827 0.31299794 -0.37236187 -0.28402755 0.38730046 0.33268583 -0.14342925 -0.389092 -0.41323915 -0.23962155 0.44499817 0.21842659 0.27842966 -0.38152382 -0.08515888 0.37871557 0.18448292 -0.32559037; -0.32263958 -0.32896644 0.4054031 -0.30170456 -0.26715264 0.1396419 0.2796748 -0.5390502 -0.12195788 -0.15681633 0.25480306 0.40946272 -0.61538565 0.005919405 -0.009088444 -0.0016941449 -0.4201208 0.16413833 -0.40230379 -0.020506253 0.056889754 -0.508778 -0.0074395635 0.16712444 -0.47190002 -0.21419246 -0.52653813 0.41798183 -0.42016912 -0.29180127 0.17139514 0.088390894 -0.2557752 0.23332259 -0.45540923 0.083172455 -0.2183393 0.12597711 -0.14586829 0.38834488 0.5097923 0.37489748 -0.25821787 0.16820237 -0.19049092 0.22630093 -0.668884 0.06945801 -0.087524615 0.19920947 0.17429172 -0.07187528 0.44654188 -0.021908937 -0.6794237 -0.306453 0.24540429 0.45665243 -0.16782162 -0.26713854 0.30155775 0.47121057 0.49773446 -0.23207904 0.007386559 -0.023787733 -0.28170764 -0.118047096 0.15260297 0.26286104 0.4139054 0.15246312 -0.23598264 0.098260224 -0.14131106 0.34133366 0.35147163 -0.406292 -0.019264385 0.49954304; 0.15382715 0.22517222 -0.04847965 0.34708834 0.17207938 -0.2293145 -0.21505156 0.1744539 -0.030741643 0.23934896 -0.3928098 -0.29449517 0.50192654 -0.15239146 -0.30697975 0.17095466 0.28665638 -0.23158692 0.35766444 -0.06762889 -0.3246002 0.5087952 0.010740499 0.013183481 0.1626226 0.14417447 0.33185652 0.08368365 0.31940714 0.31495878 0.04725253 -0.08941896 0.4854877 -0.4551802 0.0683273 -0.01883198 0.4630691 -0.018104924 0.0914822 -0.3920772 -0.34911475 -0.14498954 0.18991296 0.07585698 0.1814019 0.020354128 0.55908436 0.080633074 0.21766992 -0.24811949 -8.330077f-5 0.07710566 -0.17133677 -0.31767347 0.4874342 0.37338832 0.1682572 -0.3707083 -0.09785587 0.21136266 -0.22099113 -0.0696167 -0.079253376 0.29166448 -0.047896903 -0.09749143 0.25026956 0.16701996 0.058310356 -0.5260144 -0.18728077 -0.17625427 0.031199766 -0.11986118 0.1462857 -0.09381652 -0.078649804 0.1408885 -0.008589846 -0.3761995; 0.026267482 0.08811469 -0.29808658 0.20424126 0.2425771 -0.30199215 0.34506467 -0.49089617 0.24691866 0.2967813 -0.25887215 -0.09922518 -0.25375843 -0.4137787 -0.37757134 0.13319062 0.21475531 -0.13562647 -0.24837527 -0.48476937 -0.15457708 -0.13485603 0.38558105 0.41718945 0.040732015 0.38979945 0.06015094 0.17226772 0.41701078 -0.030005325 0.06470158 0.20015119 -0.06765464 -0.068027996 0.20788537 0.30140424 0.19221401 0.33647636 0.2617747 0.08337664 -0.22128893 -0.2711434 -0.12959939 -0.39365205 0.4519185 -0.22229932 -0.40646437 0.5989123 0.28881827 0.18028262 0.6188661 0.27143046 -0.1237956 -0.36609587 -0.5037105 0.41380078 -0.070075035 0.5143476 -0.21740496 0.350365 -0.5652231 -0.046880253 -0.036373332 0.37768692 -0.31579152 -0.3477595 0.42646718 -0.00018149549 0.03549654 -0.011745155 -0.24430512 -0.08610007 -0.112676315 0.66265607 0.36976898 -0.18435258 -0.37289274 0.24797513 0.49533743 0.11751429; -0.21602347 -0.46819603 0.14617673 -0.19286992 -0.42084688 0.35261768 0.0155737875 -0.33911076 0.08989422 -0.23053472 0.31370965 0.44207293 -0.2558718 0.1854235 0.30540347 0.024704345 -0.26845768 0.30895954 -0.13900204 0.07752052 0.4450066 -0.37234533 -0.34545603 0.09095787 -0.32425964 -0.47286388 -0.5236365 0.116533324 -0.10258588 -0.20321704 0.28874585 0.12064364 -0.20740715 0.4430159 -0.4666624 0.21950655 -0.35575163 -0.08317104 -0.31985837 0.46782345 0.29126993 0.35052893 -0.38663995 0.2304773 -0.20306589 0.19423048 0.16718094 -0.33782092 0.09609536 0.5390131 -0.19784111 -0.43891177 0.23554963 0.30405053 -0.26133674 -0.11542131 0.119899884 0.37138256 -0.21510798 -0.1433206 0.10847629 0.23177628 0.3216637 -0.11699105 0.3065205 0.17458394 -0.07991166 0.03065278 0.32974526 0.17540687 0.14442822 0.12058314 -0.26356813 -0.34421125 -0.18500666 0.4202439 0.42563605 -0.36996588 -0.35498813 0.3097702; 0.41352803 0.25142503 -0.32359338 0.15288511 0.16533545 -0.43243313 0.21516304 0.50750536 0.30910677 0.43779466 -0.17258394 -0.5195105 0.27099207 -0.19197689 -0.33357054 0.007463507 0.17800131 -0.36012936 0.4476211 -0.22298303 -0.24029322 0.6188098 0.31409687 0.13716866 0.44515148 0.5028111 0.23675044 -0.27873677 0.38739514 0.30046266 -0.49877915 -0.011830415 0.18339035 -0.43169567 0.1413412 -0.05647194 0.4139212 0.0027619514 0.20197466 -0.17793281 -0.11830457 -0.3148456 0.39641204 -0.16752674 0.25997794 -0.41587287 0.040864084 0.09846025 0.20972665 -0.4330306 0.021043427 0.36334687 -0.20465769 -0.38268137 0.32422534 0.15307845 -0.30900082 -0.20409083 -0.097706705 0.31551158 -0.38771373 -0.16608265 -0.452565 0.22035277 -0.45438075 -0.27579623 0.1799705 0.39302656 -0.22596067 -0.5060922 -0.4949976 -0.25561178 0.4269071 0.03605001 0.27555844 -0.15395617 -0.4313737 0.33612147 0.31724498 -0.45535907; 0.2370399 0.2000704 -0.2554654 0.24189197 0.17921358 -0.28433603 -0.20117643 0.37847087 0.0063523096 0.09505955 -0.49689192 -0.4427364 0.29569158 0.008721583 -0.31951872 0.28975967 0.030034564 -0.044261564 0.3583538 0.04892341 -0.15203555 0.36461627 0.3054323 0.2394903 0.13499564 0.18531598 0.28122744 -0.27471516 0.24606606 0.19983803 -0.32363763 0.048517004 0.53637147 -0.36636007 0.15110926 0.15837435 0.47474086 -0.15688832 0.1874447 -0.35877323 -0.007122746 -0.2910164 0.47609973 0.027338505 0.09956299 -0.003680232 0.14618087 0.025747957 0.18691398 -0.33132935 -0.3239101 -0.012371784 0.05680881 0.10664844 0.6126759 0.41322118 0.12743875 -0.56408674 -0.023802862 0.15211715 -0.28782475 -0.3072764 -0.34406707 0.123303466 -0.014040523 -0.3569978 0.22882977 0.21540701 -0.07669355 -0.38488802 -0.27286708 -0.10521669 0.002152075 0.061650842 0.3852121 -0.18508174 -0.36313474 0.045443222 0.16135502 -0.42217177; -0.39343926 -0.33630243 0.32539454 -0.23547255 -0.45196253 0.47642094 -0.33458182 0.18671128 -0.38755733 -0.30953786 0.37055406 0.30422035 -0.32118437 0.09332434 0.20984061 -0.2278922 -0.28166518 0.30682927 -0.21774861 0.19084154 0.19384165 -0.29625928 -0.3222155 -0.2822982 -0.43409204 -0.2837058 -0.13814795 0.07342474 -0.19099024 -0.2106867 0.1399796 -0.43237337 -0.49936244 0.18665987 -0.11550267 -0.342691 -0.1732738 -0.19051312 -0.16685101 0.43274626 0.2380875 0.4757966 -0.28547683 0.3481944 -0.4123342 0.1864287 0.5767437 -0.23724811 -0.24077222 0.23697558 -0.24923249 -0.19148532 0.36334983 0.18712293 -0.28459248 -0.447147 0.23616916 0.23038882 0.36969575 -0.3967341 0.20200586 0.33857116 0.26234466 -0.26789808 0.3557261 0.39193106 -0.17732581 -0.09403819 0.3823295 0.19564857 0.34929788 0.55253994 -0.34880674 -0.43068027 -0.44164637 0.40101746 0.3344899 -0.47920284 -0.37080544 0.3290472; 0.0039822794 -0.16750984 0.42307383 -0.34638792 -0.26284102 0.4939657 -0.42400363 0.05968077 -0.16909267 -0.2814408 0.4116196 0.22658873 -0.17849177 0.30427808 0.17147109 -0.3148431 -0.39472485 0.28397685 -0.1871163 0.28975695 0.1850362 -0.3661891 -0.41918522 -0.41091588 -0.23320872 -0.4884998 0.033197526 -0.1991743 -0.43506745 -0.34500527 0.23306829 -0.44926068 -0.4757213 0.15528591 -0.16982232 -0.26800254 -0.2906294 -0.2678146 -0.29579172 0.20379686 0.46808973 0.3039357 -0.25462925 0.16551358 -0.38631123 0.26927325 0.06683199 -0.22102064 -0.29035714 -0.034981593 -0.46961924 -0.15886009 0.18220085 0.3555308 -0.051960196 -0.44326824 0.06071416 0.22861739 0.46847993 -0.4864257 0.4207112 0.16475138 0.026218211 -0.3208991 0.21112342 0.38320488 -0.33556148 -0.2890131 0.13655894 0.49534845 0.2206138 0.15660545 0.0027601905 -0.3276107 -0.3509044 0.08278587 0.5312275 -0.52492833 -0.2967556 0.2530038; 0.06748737 0.36268187 -0.078215584 0.29739216 0.009053852 -0.45772144 0.027488826 0.22834992 0.47647786 0.122260556 -0.49340245 -0.21400069 0.5151971 -0.22311044 -0.43209276 0.490607 0.3173179 -0.103787236 0.37324157 -0.15254283 -0.4541316 0.5291873 0.32417476 0.41796857 0.3263628 0.14073588 -0.01052186 -0.22585014 0.5046463 0.11979604 0.04313114 0.15240417 0.54846114 -0.13313839 0.1529889 0.313156 0.399671 0.21653834 0.10539375 -0.2230446 -0.23408331 -0.3183114 0.68452823 -0.11367538 0.10484934 0.02839732 0.010157925 -0.01660691 0.26989245 -0.07546199 -0.12215005 0.21688887 -0.06425649 -0.34134987 0.66699386 0.6493269 0.013813856 -0.55561155 -0.08601986 0.10368879 -0.29040796 -0.10763058 -0.040181976 0.53156775 -0.0466683 -0.6227558 0.47168726 0.47201076 -0.07755213 -0.71329343 -0.21168 -0.14372157 0.1749125 -0.02793087 0.29254004 -0.069640435 -0.56269884 0.2079431 0.11715854 -0.3429122; 0.50271946 0.20837879 -0.45203435 0.27288637 0.23194483 -0.46435556 0.12778038 0.39733598 0.24669383 0.15953186 -0.3854594 -0.29619 0.2678406 -0.1326603 -0.5092597 0.23018147 0.4266306 -0.21744305 0.39850292 -0.24462475 -0.3259758 0.28885862 0.38342634 0.15310307 0.2973454 0.14621654 0.43223482 -0.61659247 0.31047687 0.38248336 -0.22427864 0.16070543 0.2673726 -0.4561716 0.31131986 0.17817463 0.26786587 0.16187133 0.12654819 -0.4067939 -0.18415079 -0.3886209 0.2854157 -0.17277806 0.22860695 -0.14999449 -0.1120841 0.399077 0.28712356 -0.40172473 0.0040760427 0.13066319 -0.1048688 -0.36281878 0.5440966 0.29283264 -0.4665769 -0.3647986 -0.106425315 0.09287581 -0.31377742 -0.32985568 -0.5037076 0.2615584 -0.14085802 -0.46509215 0.23325272 0.39463574 -0.1737454 -0.22468196 -0.14047493 -0.4008593 0.28198305 0.080440514 0.40140483 -0.4000454 -0.47818404 0.3478278 0.33321217 -0.31274614; -0.018611623 -0.046026304 0.29626584 -0.3351084 -0.20327711 0.32226148 -0.47620222 0.4263282 -0.13807295 -0.45490906 0.21625553 0.20593803 -0.30070105 0.36098337 0.28619066 -0.36694703 -0.38778514 0.23811859 -0.2091332 0.36672786 0.43854207 -0.09546804 -0.22890295 -0.18552501 -0.2740974 -0.15996617 -0.16178593 -0.07856882 -0.29404587 -0.36732587 0.21646713 -0.33093327 -0.10471095 0.3631451 -0.41104624 -0.33469012 -0.37103954 -0.24899365 -0.38616887 0.16937597 0.13434376 0.47912437 -0.15421571 0.3636097 -0.15898472 0.37468582 0.57320774 -0.50720704 -0.4802961 -0.013885851 -0.3588112 -0.44547033 0.12727162 0.20820405 0.058186356 -0.49313897 0.29746166 0.21629836 0.33346266 -0.36512905 0.43555725 0.24358492 0.13500458 -0.267471 0.33442032 0.44268566 -0.2507341 -0.49373952 0.093060255 0.4666246 0.31345737 0.3069415 0.10998874 -0.2475821 -0.40448153 0.16631265 0.44060007 -0.31843013 -0.20595215 0.25516328; -0.41980717 -0.47881612 0.22622813 -0.077847265 -0.41355744 0.6023699 -0.30228984 -0.54982084 -0.3821409 -0.20973177 0.44347554 0.5911317 -0.67766076 0.23837003 0.51101315 -0.5209996 -0.39544675 0.43979278 -0.605939 0.12127771 0.46475458 -0.3937427 -0.5712773 -0.25468466 -0.32738817 -0.45914495 -0.21388347 0.16548955 -0.50585973 -0.3301978 0.2142233 -0.18141001 -0.340044 0.54071814 -0.24699278 -0.08588776 -0.3636513 -0.36757606 -0.42825434 0.43759292 0.42149672 0.38929138 -0.44785473 0.115761265 -0.19257838 0.050953757 0.10198563 -0.10484985 -0.27732018 0.117448606 0.13087428 -0.27531976 0.09847998 0.26924455 -0.35934216 -0.41853198 0.16410251 0.55352247 0.19183332 -0.41360426 0.31102708 0.3427156 0.41322744 -0.55143225 0.13754159 0.3663851 -0.43733665 -0.56542313 0.2391667 0.51792765 0.42888674 0.3513064 -0.2255209 -0.11318668 -0.5394522 0.16643149 0.5001311 -0.2929326 -0.20650558 0.26176515; 0.23983736 0.22335061 -0.44586873 0.32913575 0.24641794 -0.24211894 0.37330636 -0.011708449 0.28630635 0.1450045 -0.34988144 -0.20154768 0.26479596 -0.23526536 -0.17748317 0.31415752 0.18612677 -0.23496877 0.4255941 -0.27196783 -0.28927895 0.50146115 0.4711592 0.41284484 0.40874767 0.49461317 0.2851116 0.0116387475 0.2313669 0.46980914 -0.2157224 0.2426371 0.33821946 -0.30367294 0.30728418 0.24536969 0.33855298 0.31066772 0.28531992 -0.5277388 -0.2736413 -0.49966723 0.35060102 -0.46048984 0.21159434 -0.24756077 -0.35938466 0.11783283 0.37662962 -0.27296144 0.19550867 0.18030255 -0.4610103 -0.35463014 0.38587022 0.2882196 -0.26189023 -0.19428955 -0.36509803 0.17959252 -0.3391932 -0.2592304 -0.20939638 0.34240216 -0.318917 -0.11569033 0.1963358 0.38189024 -0.25340056 -0.22091652 -0.25310808 -0.17481492 0.2284277 0.29354003 0.3844611 -0.31641778 -0.41934848 0.14586274 0.40395913 -0.29420242; -0.43422964 -0.20932353 0.23855878 -0.17005725 -0.40949088 0.17716202 0.13599676 -0.34743097 0.013411436 -0.47222954 0.08302241 0.47639096 -0.3943078 0.06117127 0.3374063 -0.002643895 -0.49794084 0.12586339 -0.16323572 0.22999762 0.2352953 -0.53353155 0.016186824 -0.10447731 -0.4531201 -0.14387771 -0.36834213 0.16076802 -0.27503166 -0.31192827 0.514393 0.059621803 -0.2691153 0.47217074 -0.387266 0.24153319 -0.1869561 0.13684885 -0.046387132 0.33937398 0.17994821 0.30966002 -0.26277316 0.42167723 -0.29616368 0.4762386 -0.05044536 -0.08135789 0.23881568 0.341845 0.2409075 -0.31707585 0.4842723 0.40962034 -0.45886204 -0.32654953 0.44665414 0.35444564 0.022757746 -0.28296703 0.15152839 0.35548574 0.21747181 0.011461559 0.13063742 0.030969758 -0.14805263 0.00866096 0.31887394 0.2276543 0.45959792 0.31614766 -0.46804035 -0.2765661 0.043633044 0.20351131 0.06879177 -0.35082802 -0.37272245 0.10059076; -0.19183718 -0.45744383 0.20128103 -0.24075393 -0.094420224 0.1821477 -0.12324883 -0.5347583 -0.008384194 -0.13323797 0.2933921 0.46290287 -0.52059025 0.22245723 0.24111979 -0.17852251 -0.13101873 0.15279646 -0.22263065 0.25459006 0.11701929 -0.23521423 -0.25919515 -0.22066346 -0.15903935 -0.22176716 -0.20736678 0.007296693 -0.17685361 -0.38857314 0.18817605 0.102818616 -0.43645033 0.071877286 -0.100996226 -0.17438245 -0.10366934 0.0758299 -0.22574145 0.37432718 0.30101055 0.43419152 -0.4660759 0.44867343 -0.097841784 0.28214398 0.039191503 -0.10212651 -0.25433588 0.2370607 0.019693121 -0.33503184 0.08896694 0.1477363 -0.49578956 -0.2894802 0.34458297 0.30962703 -0.021761995 -0.052183088 0.35348192 0.15274207 0.4747188 -0.34332982 0.07564743 0.2421104 -0.17135207 -0.30779728 0.14072372 0.44406322 0.22950369 0.28915808 -0.3446142 -0.0459012 -0.36445227 0.35948688 0.33175498 -0.290677 -0.0588403 0.23283564; -0.14081386 -0.21508652 0.2315748 -0.46876958 -0.205982 0.28843117 -0.48802313 0.3308038 -0.16255303 -0.3020429 0.36470094 0.1715389 -0.12726519 0.28718367 0.21750776 -0.30776548 -0.18418005 0.2030228 -0.24992448 0.31308606 0.21187042 -0.01869603 -0.34027454 -0.20604292 -0.22247963 -0.37760088 -0.032582726 -0.42374852 -0.16476429 -0.156998 -0.111775175 -0.15529546 -0.43645248 0.26767915 -0.4883632 -0.23462595 -0.19580972 -0.3551591 -0.30453938 0.20679504 0.20239998 0.33958292 -0.42713118 0.46859944 -0.23890877 0.18980953 0.39178544 -0.45828733 -0.3797202 -0.20125383 -0.41711596 -0.43401483 0.18836105 0.36560103 -0.0022820442 -0.14448895 -0.01687213 -0.111725785 0.30760595 -0.46544343 0.3202628 0.26236022 0.10549194 -0.2607934 0.23382385 0.363215 -0.48143253 -0.17364886 -0.016900511 0.45056722 0.36089668 0.18761644 0.12491724 -0.383415 -0.17144446 0.24164237 0.3032859 -0.2096033 -0.4034617 0.17042522; -0.45881024 -0.51208895 0.40905866 -0.39421332 -0.3471076 0.3370784 0.13076644 -0.7056567 -0.28792965 -0.11157249 0.2808488 0.37201685 -0.7215444 0.26811567 0.10307352 -0.14686614 -0.35271174 0.25167385 -0.3293386 0.25881994 0.27476275 -0.38046739 -0.12935223 0.013089471 -0.2988719 -0.26707143 -0.5418186 0.5231463 -0.34450638 -0.42349723 0.25105023 -0.1320328 -0.5729111 0.5974292 -0.114181615 0.062067743 -0.06761544 0.16787572 -0.077665545 0.5544421 0.38171673 0.1849293 -0.36409387 0.09478204 -0.28356287 0.51160204 -0.49385208 0.105425656 -0.09850363 0.2969391 0.18087712 -0.0029950985 0.2848171 0.1639408 -0.41235325 -0.17009456 0.01599262 0.26319167 -0.12879139 -0.3825656 0.117101 0.24099389 0.44660643 -0.086770885 -0.1288668 -0.12512386 -0.155559 -0.19083956 0.3602194 0.4529856 0.35170493 0.61659837 -0.4721191 0.08912769 -0.2919588 0.20466237 0.34556308 -0.24882096 -0.21431865 0.31204683; -0.009378173 0.53629476 0.05325081 0.020059511 0.01771633 -0.31144372 -0.35654297 0.3888163 0.30394512 0.1695519 -0.16897525 -0.60546577 0.3319819 0.11576974 -0.33695403 0.13538982 0.0007277115 -0.24641056 0.44797966 0.30722982 -0.5279385 0.49456325 -0.020551573 0.15798493 0.34803852 0.05507422 0.06250827 -0.29663908 0.5926531 0.3143889 -0.06319973 0.018509567 0.5303368 -0.3780123 0.06872779 -0.008568965 0.31613797 -0.020138044 -0.37761343 -0.565023 0.011361466 0.082975425 0.70685655 0.16554421 -0.19094224 0.07014593 0.4432656 -0.2746614 0.05662909 0.09654137 -0.4954887 -0.2091475 -0.0798835 0.18320395 0.7043093 0.6609025 -0.041354936 -0.74589676 0.05793188 -0.15009722 0.016082766 0.06044269 -0.17879617 0.027973771 0.094034664 -0.38454163 0.5813415 0.43492332 0.065473385 -0.84192985 -0.046857152 -0.51227456 0.19786018 -0.36495328 0.13866284 -0.058138885 -0.5476036 -0.07658637 -0.2725497 -0.5585022; 0.15567656 0.009700621 0.17266652 -0.32302535 -0.21282744 0.36961 -0.40252322 0.42183843 -0.3965951 -0.14891888 0.16179083 0.1869638 0.09130299 0.34899604 0.18444225 -0.1501258 -0.33989576 0.23912926 -0.06214192 0.37518743 0.34528336 0.24368359 -0.5230274 -0.33805877 -0.28188786 -0.2644149 -0.2452507 -0.44637346 -0.2837503 -0.22966829 0.12853135 -0.3476415 -0.0044469163 0.16076382 -0.5106451 -0.2267365 -0.45083237 -0.42965856 -0.3949508 0.28773072 0.39958265 0.24535222 -0.08308891 0.2883146 -0.33434895 0.13838461 0.41500378 -0.48843297 -0.16458964 0.065371305 -0.46153647 -0.5590653 0.15955469 0.535477 0.26557362 -0.32804945 0.21274526 -0.14620629 0.5128386 -0.3834337 0.51515514 0.41322127 0.023230268 -0.3188397 0.3881249 0.37459892 -0.11628523 -0.37963408 0.094559036 -0.025272243 0.5230441 0.24685568 -0.058108393 -0.28521466 -0.43662003 0.13724849 0.31896615 -0.5043947 -0.34166726 0.24795432; -0.27493957 -0.32554463 0.44702336 -0.2886947 -0.2680614 0.1819283 -0.14105342 -0.11754613 -0.43395266 -0.15326503 0.25927898 0.36046928 -0.41375533 0.12911318 0.4590796 -0.29238746 -0.29195172 0.4047675 -0.1406001 0.123658195 0.3514588 -0.29130486 -0.27667278 -0.45971736 -0.44986463 -0.44021824 -0.4544071 0.15985619 -0.43799084 -0.42030656 0.41523743 -0.38973272 -0.15782373 0.3858239 -0.11764133 -0.13604513 -0.36749622 -0.32036132 -0.15142007 0.483706 0.12685065 0.12022177 -0.3942868 0.23316047 -0.27339566 0.46649963 0.24584033 -0.102493316 -0.12881306 0.38966447 -0.25952366 -0.13830312 0.34768194 0.24804163 -0.31645665 -0.34785387 0.21900049 0.21863006 0.3186081 -0.14214239 0.3128337 0.3215081 0.15578087 -0.49356884 0.24655333 0.2755413 -0.3811421 -0.10110073 0.26321095 0.35581633 0.45270848 0.49793917 -0.5092897 -0.35036415 -0.2773848 0.25774342 0.31663233 -0.35219923 -0.23660982 0.24686567; -0.34085834 -0.46790564 0.36959165 -0.16308433 -0.24263287 0.02377497 0.19109024 -0.43534818 0.074767105 -0.12465285 -0.19718336 0.23030736 -0.30045065 -0.027105263 0.032970555 0.38958573 -0.37794724 0.3690624 -0.14070736 -0.028686214 0.2440074 -0.24177335 0.24560115 0.16110007 -0.19249846 -0.2884918 -0.4179308 0.43349534 -0.044754665 -0.24042708 0.24241096 0.12134227 -0.40922108 0.13941045 -0.2537425 0.40721413 0.20664026 0.28446114 -0.15906526 0.3473918 0.33742425 0.16203362 -0.19154437 0.3937656 0.0907431 0.24805807 -0.22604905 -0.17551182 0.46071193 0.24397318 0.44912985 -0.061155595 0.40136933 0.037830368 -0.41372335 -0.13248439 0.20510934 0.2562609 -0.34094173 -0.2001255 0.10310416 0.27505994 0.40583214 0.22957815 0.02010271 -0.17300852 -0.11769271 -0.14267516 0.13013208 0.16659719 0.40575042 0.28428432 -0.4862839 0.16251975 0.22140689 0.17552908 0.12931174 -0.43883365 0.007978115 0.06567424; 0.2628376 0.44432136 -0.26798144 0.16814853 0.44998455 -0.27482578 0.16787575 0.18625191 0.32729748 0.503085 -0.43693548 -0.2406465 0.34848422 -0.42374504 -0.17964636 0.4870023 0.23656316 -0.19810948 0.31899095 -0.13897304 -0.43048355 0.18578865 0.23689546 0.10044827 0.16443783 0.4291071 0.4948015 -0.22782561 0.3374562 0.15458892 -0.43158728 0.42676622 0.39331943 -0.15460193 0.14241727 0.105222434 0.33603606 0.08851005 0.30319336 -0.30051032 -0.15928602 -0.33879223 0.29504976 -0.3782864 0.33593714 -0.21337748 -0.18061927 0.11964755 0.12281301 -0.31567854 0.35549992 0.10984661 -0.20235518 -0.10968271 0.29963443 0.36276883 -0.44432104 -0.39063781 -0.1658122 0.12724006 -0.13216537 -0.16694547 -0.22542956 0.370604 -0.2584527 -0.26672885 0.12606241 0.19377483 -0.38494667 -0.43443394 -0.48668528 -0.4682074 0.3951741 0.27724087 0.19257134 -0.3040741 -0.38236782 0.21698682 0.45832643 -0.20654656; 0.109659135 0.058441363 -0.2862207 0.36331165 0.3999959 -0.4884282 0.5696982 -0.49184754 0.13606809 0.4834008 -0.40799463 -0.3492153 0.1973812 -0.47198045 -0.39421853 0.29596895 0.13579759 -0.35638607 0.23902422 -0.5279143 -0.22623177 0.33228523 0.45115128 0.40230212 0.2296192 0.48083732 0.3193932 0.14577885 0.19536924 0.27609563 -0.027582703 0.19154972 0.27706632 -0.24137732 0.294671 0.21365695 0.36062816 0.24986723 0.32510632 -0.30078664 -0.42834532 -0.52322596 0.3907746 -0.37583688 0.4634629 -0.39407766 -0.14601192 0.21593386 0.44394398 0.13364062 0.4608636 0.44462258 -0.21672457 -0.37277177 0.13316055 0.45658296 0.010556455 -0.17408666 -0.4614447 0.5415632 -0.32453662 -0.18186578 -0.22281937 0.48267332 -0.07263667 -0.32900307 0.13154685 0.08090614 -0.18181215 -0.07817485 -0.22032127 -0.49281296 -0.0632863 0.4321533 0.19167903 -0.054375403 -0.10298176 0.4815181 0.22837736 -0.16048633; 0.32490316 0.50981426 -0.16067943 0.21406728 0.39576447 -0.228803 -0.28040552 0.7606831 0.13801277 0.39093113 -0.42365426 -0.47647825 0.5687473 -0.085840724 -0.5183217 0.06562703 0.2832343 -0.16383372 0.5446868 -0.056049 -0.48839703 0.6518333 0.09091687 -0.0009546322 0.60781467 0.07354002 0.40148336 -0.5181029 0.34172356 0.2812755 -0.07573827 -0.23099929 0.59789467 -0.35872427 0.10698125 0.036492024 0.41801184 -0.10594693 -0.13131389 -0.6828708 -0.41699025 -0.36463892 0.37530038 0.044483572 0.1798704 -0.19246392 0.7346265 -0.23714632 -0.04671696 -0.32696918 -0.46230227 -0.051793844 -0.34333736 0.13522677 0.71234614 0.55564106 0.20392476 -0.76795655 0.0044449046 0.25919306 0.050689016 -0.3741111 -0.26210445 0.17827421 0.12213711 -0.15253621 0.24781696 0.36680457 0.036468364 -0.70126647 -0.092111245 -0.47630018 0.23333219 -0.16628398 0.21292546 0.13729472 -0.5414761 0.39596343 -0.22248976 -0.60118604; 0.029585535 -0.08129276 0.447641 -0.5293708 -0.5207486 0.36408103 -0.30707535 0.5317132 -0.21867764 -0.38135332 0.31038705 0.1826816 -0.0801319 0.38713768 0.23882583 -0.4020809 -0.24267751 0.5058845 -0.08126344 0.2626342 0.32823732 0.008641835 -0.27506065 -0.24113023 -0.103850156 -0.43490303 -0.12025997 -0.071387604 -0.3590638 -0.217674 0.14040835 -0.110611595 -0.23728009 0.21599454 -0.28573656 -0.1745391 -0.23575383 -0.40188596 -0.22417021 0.17552131 0.14327751 0.34902537 -0.115172744 0.13453966 -0.17948326 0.25878388 0.50448734 -0.36124012 -0.3471067 0.03326925 -0.30025044 -0.3205084 0.15730795 0.19632322 -0.049919475 -0.26975182 0.23316242 -0.03305177 0.22148505 -0.17755382 0.20375344 0.1410319 0.10734934 -0.41390184 0.43167296 0.29813027 -0.2328113 -0.14732225 0.09654324 0.22605385 0.34557596 0.15639539 0.024468973 -0.27153206 -0.23864707 0.27350125 0.33455703 -0.1693703 -0.51366293 0.41931188; -0.35425276 -0.45585418 0.27535188 -0.27704823 -0.2871568 0.089687034 -0.33361143 0.06226401 -0.17556362 -0.32316458 0.22036591 0.164178 0.021421552 0.16344377 0.31259066 -0.027441371 -0.4438367 0.4239165 -0.373842 0.15021086 0.15413243 -0.11952109 -0.42761046 -0.18602785 -0.14774331 -0.121335514 -0.21289314 0.08687919 -0.17243618 -0.31221378 0.44100833 -0.23491998 -0.067083545 0.36550453 -0.18566191 -0.18335544 -0.098998405 -0.09011413 -0.43210167 0.36315215 0.43795055 0.25770792 -0.39937943 0.5153873 -0.21088335 0.23442568 0.41655624 -0.21213038 -0.2645948 0.40356255 -0.26474598 -0.26471674 0.39197856 0.14181146 -0.08637696 -0.3275302 0.47667786 0.20748876 0.32865524 -0.31602848 0.41181433 0.36446077 0.47752696 -0.38637793 0.4379835 0.10723324 -0.07170476 -0.31048176 0.123710945 0.24893597 0.30178994 0.2716446 -0.33636892 -0.43894 -0.12864001 0.44994637 0.3539924 -0.20342313 -0.5328616 0.10556216; 0.071254216 0.6192697 -0.2016362 -0.2979582 -0.42867738 -0.050133407 -0.65625745 0.81231695 -0.08954597 0.1805325 0.1338752 -0.861377 0.6248176 0.55101115 0.122582994 -0.09465373 0.124299034 -0.14010322 0.6720729 0.71922964 -0.13138989 0.5196618 -0.023094451 -0.06677189 1.0064938 -0.007969971 0.44230038 -0.14421609 0.3474622 0.26173759 -0.283505 -0.117343545 0.69337493 -0.5307629 -0.42454448 0.007111475 -0.21652389 -0.35057572 -0.44752935 -0.7961455 -0.019712623 0.51779765 0.46308285 0.4544786 -0.6061487 0.012274252 0.6687453 -0.69022024 -0.2420638 -0.08556929 -0.46716538 -0.5673335 -0.10043311 0.6220182 0.5406976 0.35735053 0.07128742 -0.57664996 0.0750525 -0.53401506 0.6276309 -0.1302877 -0.42343387 -0.13429748 0.2202744 0.08791794 -0.09268749 0.16894007 -0.037047047 -0.44473767 -0.06510606 -0.7257225 0.14924543 -0.58815604 -0.1868324 -0.049291383 -0.24632303 0.14299844 -0.29318902 -0.63521796; 0.17486665 0.3771473 -0.3017571 0.23274948 0.34985694 -0.24666134 -0.12570713 0.82876647 -0.121300764 0.41084257 -0.275163 -0.54788417 0.70284307 -0.0059231673 -0.04423716 0.003204289 0.20573466 -0.45116335 0.5400214 -0.11373178 -0.279871 0.571152 0.10625879 -0.41335464 0.56517625 0.46275067 0.51258004 -0.35718885 0.28440732 0.25547397 -0.48235467 -0.07696729 0.6303613 -0.63279814 0.15692075 -0.42344758 0.110411115 -0.12335428 0.03533347 -0.68812436 -0.47020778 -0.2525333 0.23974316 -0.070712425 0.014074385 -0.18386383 0.80295134 -0.19074424 -0.105098814 -0.2985564 -0.5170307 0.13710189 -0.3549195 0.17406811 0.68069226 0.25719342 0.052753784 -0.62597096 0.38570434 0.32970878 -0.07399284 -0.37515652 -0.36693114 -0.033730756 0.17456089 0.2354165 0.1327454 0.07507427 -0.1222959 -0.45933503 -0.32460424 -0.57062453 0.35399926 -0.3410845 -0.09665556 -0.0056031407 -0.30850315 0.2669513 -0.21435332 -0.5057035; 0.089093864 -0.07777428 -0.10466582 0.4920692 0.18005595 -0.40637338 0.51797295 -0.46188214 0.4531805 0.2892009 -0.28757796 0.11248514 -0.19813819 -0.37839296 -0.5255352 0.17799616 0.12708916 -0.25444812 0.110560864 -0.417606 -0.1843359 -0.33001384 0.26079193 0.4317539 0.178933 0.37358817 -0.0662398 0.20917468 0.4996109 -0.041519284 -0.009615909 0.3707777 -0.020609776 0.0473399 0.4719817 0.34588033 0.17373858 0.2683281 0.49611992 0.11587314 -0.36231226 -0.2948934 0.38278592 -0.16327512 0.24049981 -0.20678249 -0.39147195 0.48973706 0.40667087 0.23990643 0.5884867 0.2987727 -0.11392207 -0.24701922 -0.44258258 0.1682669 -0.29269785 0.0043658507 -0.14390534 0.53540415 -0.54709667 -0.07546832 0.05771513 0.20775054 -0.2839981 -0.36372146 0.24413773 0.14160971 0.03267736 -0.056248385 -0.09273163 -0.022449158 -0.34817848 0.6498152 0.53515786 -0.049926765 -0.40200245 0.19072905 0.32394913 -0.37138924; 0.33193892 0.103590615 -0.14641434 -0.06111867 0.038744487 0.45758954 -0.20668401 0.5870524 -0.22323811 -0.18994384 0.47527558 -0.01524357 0.12106593 0.4561217 0.51283365 -0.2165332 -0.044577423 -0.04054583 0.21017948 0.116909936 0.45282614 0.20042272 -0.38584483 -0.4426672 -0.06434863 -0.32080352 0.42208967 -0.36427465 -0.45595872 0.42269164 -0.4158998 -0.14942293 -0.24088576 0.036107965 -0.5009377 -0.4330493 -0.33138198 -0.37333617 -0.49487796 -0.33998844 0.3312692 0.30302396 -0.2190801 -0.03447691 -0.49007136 -0.13253225 0.3401965 -0.31732437 -0.40795654 -0.52994466 -0.038773082 -0.46344167 -0.13581988 0.45713875 0.24063212 -0.61616415 -0.2696827 -0.1463302 0.20290105 -0.28714612 0.14873633 -0.08371967 -0.44699156 -0.26843256 0.02733409 0.5470423 -0.5142609 -0.30953178 -0.32520545 0.13027926 0.14670883 0.09932998 0.1916486 -0.16917425 -0.29512113 -0.018828899 0.669138 -0.36953142 0.057226174 0.25667927; -0.058077037 -0.29304403 0.16545859 -0.44466633 -0.2496658 0.5184077 -0.40792796 -0.09489162 -0.33112618 -0.39611167 0.32489932 0.21736018 -0.1417141 0.41454962 0.3725204 -0.10168543 -0.48472622 0.40325 -0.22954364 0.48686796 0.4480942 -0.3289969 -0.43407413 -0.4451206 -0.23695959 -0.52093244 -0.23224142 -0.11695114 -0.43298984 -0.2674802 0.046919517 -0.43779436 -0.459499 0.21808746 -0.37395415 -0.14372283 -0.29760262 -0.32866925 -0.40100124 0.32397637 0.16844197 0.30143538 -0.48323935 0.37265173 -0.14893867 0.47934866 0.400766 -0.12008949 -0.19286196 0.3534163 -0.21456772 -0.2793488 0.23872125 0.25315475 -0.13264969 -0.22659309 0.029456329 0.2897889 0.4491274 -0.27779806 0.2826241 0.37697586 0.12510914 -0.35795677 0.10228119 0.1687766 -0.34006175 -0.35958794 0.07913 0.41438112 0.4205973 0.20605132 -0.1473419 -0.40996706 -0.21671519 0.36817974 0.4327751 -0.51571286 -0.18039306 0.46470803; 0.034004066 -0.2367516 -0.28628725 0.14641328 0.12662314 -0.48200473 0.23816565 -0.608861 0.13553676 0.34929612 -0.55589765 -0.18633623 0.10684073 -0.2843516 -0.352826 0.30615866 0.18386023 -0.27391875 -0.023840576 -0.5135656 -0.42905432 -0.051266976 0.48415345 0.14355871 -0.08701656 0.3924638 -0.10603005 0.471153 0.35714713 -0.10766914 0.42843094 0.4021575 0.05553378 -0.11051555 0.40153456 0.2350542 0.49387065 0.29330507 0.30685 -0.20752254 -0.3019243 -0.20039865 0.20097812 -0.16526392 0.5353918 -0.053558487 -0.34176326 0.14832702 0.1170302 0.15998639 0.5114733 0.49794927 -0.010872273 -0.39043388 -0.22110206 0.4677463 0.21559006 -0.006069866 -0.08029801 0.33239502 -0.14859198 -0.2412293 0.28020057 0.12638888 -0.14013524 -0.24777386 0.21657708 0.37874952 0.1814885 -0.18530898 -0.36536798 -0.1249257 -0.21282028 0.053976137 0.34848046 -0.04368804 -0.5490491 0.43691978 0.113232665 -0.45959288; -0.04483344 -0.53424233 0.23605582 -0.24965072 -0.23697838 0.015031442 0.23908207 -0.5039203 -0.20853059 -0.11627159 0.22683339 0.5038771 -0.4818994 -0.04071642 -0.021298897 0.08419666 0.051920317 0.23803945 -0.36507958 -0.30879006 0.19460858 -0.47810072 -0.123507984 0.23143707 -0.38105735 -0.070454806 -0.20005623 0.34255996 -0.31236577 -0.13052437 0.1004343 0.17429626 -0.29220396 0.11845911 0.06545489 -0.028163275 -0.3345224 0.16522422 0.10526415 0.3486873 0.1477024 -0.07003686 -0.5193874 -0.019857539 -0.060969327 0.09101035 -0.45305985 0.15268382 0.31567737 0.016502842 0.5664937 0.29871282 -0.12517296 -0.40150642 -0.56259996 -0.56542677 -0.045677666 0.6067098 -0.08151333 0.17140912 -0.2256726 0.017977143 0.024568751 0.067613095 -0.060824394 0.12314964 -0.3678988 -0.19460265 0.13267168 0.44244727 0.15834945 0.3860515 0.009831718 0.32230452 -0.34258616 0.0068637184 0.50848854 -0.11784284 0.029056229 0.48964334; -0.54469246 -0.25345123 0.15172112 -0.50498223 -0.40446052 0.019168144 -0.26660338 -0.22680755 -0.080636494 -0.16893157 0.18056175 0.35090008 -0.040455356 0.066758364 0.027392177 0.1543188 -0.24939874 0.26643825 -0.24380487 0.15595809 0.03431795 -0.30916905 -0.2538526 0.15791917 -0.40495715 -0.24074389 -0.29648757 0.15772597 -0.37961626 -0.45443118 0.2240784 0.13547608 -0.39089072 0.19478096 -0.25787222 0.24445702 -0.047513265 0.2254329 -0.4799153 0.27834034 0.15810236 0.16055901 -0.16851386 0.3183558 -0.27361313 0.52749735 0.22182094 -0.24325319 -0.01955976 0.21955815 -0.13094777 -0.34009913 0.2601381 0.16496119 -0.14814433 -0.079939045 0.17993727 0.43446657 0.071929425 -0.223879 0.1740838 0.27481052 0.5530279 -0.24709217 0.37299022 0.06586427 -0.33474636 -0.18541831 0.2285538 0.23094988 0.35008582 0.20387913 -0.2822595 -0.043801624 0.028949188 0.29732537 0.010989802 -0.3802566 -0.14861257 0.19246964; 0.14598589 0.124347486 -0.11434362 0.43344715 0.3129256 -0.22300929 0.33127576 -0.4995218 0.057521373 0.26520228 -0.20644283 -0.15218405 -0.304967 -0.5843702 -0.35994074 0.29351524 0.21209183 -0.48591688 -0.025388418 -0.3152084 -0.24932896 -0.19622213 0.2428102 0.14905187 0.20279656 0.31891114 0.28243235 0.41788113 0.08591721 -0.0027312767 -0.26241624 0.111487955 0.00935592 -0.31602716 0.26416615 0.21244872 0.2850025 0.4202525 0.5960624 -0.2177012 -0.39972076 -0.43147135 0.0875293 -0.4275641 0.21675646 -0.21060257 -0.5857005 0.39549977 0.4347197 0.026557991 0.4735059 0.32945585 -0.5152025 -0.62831914 -0.29323158 0.13681455 -0.3224586 0.07512486 -0.20774823 0.5445602 -0.38502622 -0.4466417 -0.35397425 0.10858559 -0.45026764 -0.2786109 0.09950643 0.2154186 -0.024034636 -0.15615547 -0.16675825 -0.29822016 0.23512252 0.5255773 0.28943166 -0.25376564 -0.28887603 0.2697743 0.38402188 -0.0025821708; -0.12625547 -0.32261148 0.44045782 -0.2126904 -0.40573612 0.48474455 -0.15885174 -0.404708 -0.29390985 -0.38130987 0.4663852 0.51964915 -0.26543543 0.3139477 0.4765367 -0.43545473 -0.279256 0.46098036 -0.22896233 0.34972847 0.120585464 -0.29541346 -0.31844527 -0.20203711 -0.42182648 -0.3881475 -0.18603756 -0.12505849 -0.20951962 -0.21399608 0.17677467 -0.1695436 -0.26286283 0.2435754 -0.1643209 -0.26268804 -0.42094412 -0.15460612 -0.39292723 0.21728383 0.129253 0.4350442 -0.370135 0.20882782 -0.2515373 0.26529744 0.010557269 -0.3288691 -0.1697443 0.42947572 -0.22640066 -0.4018492 0.25229177 0.18832 -0.42506567 -0.44691756 0.10015638 0.25112092 0.3997984 -0.28380418 0.35182574 0.44208866 0.22237812 -0.22506545 0.18689463 0.25233597 -0.4728752 -0.3586755 0.36444992 0.20801066 0.15852375 0.37253448 -0.50416577 -0.19035694 -0.29055828 0.33767396 0.3914621 -0.2202709 -0.11160436 0.22810422; -0.23672421 -0.36311558 0.5002467 -0.24684447 -0.21890922 0.39194867 -0.30127576 0.30987233 -0.43828595 -0.19938746 0.2324601 0.30028012 0.07161476 0.3580764 0.4079076 -0.4084216 -0.1864913 0.13340054 -0.15380701 0.4608025 0.42367336 0.010811623 -0.1732823 -0.1453775 -0.37077525 -0.17131867 -0.1752386 -0.232058 -0.3325785 -0.3733463 0.09235755 -0.17230728 -0.12734133 0.18633386 -0.17597543 -0.3582173 -0.14058106 -0.16029009 -0.29042816 0.32724446 0.29878768 0.2630131 -0.041907202 0.36580762 -0.19716449 0.08617493 0.5225156 -0.53571993 -0.15157512 0.19129863 -0.41093 -0.45125204 0.20749155 0.19524583 0.06516655 -0.18521225 0.2090951 0.14914991 0.16476476 -0.4400082 0.49056458 0.14099716 0.27627033 -0.16283129 0.2543489 0.30717066 -0.34889406 -0.18082921 0.03193391 0.017257335 0.33051383 0.23715341 -0.15427572 -0.2862014 -0.31682184 0.2546514 0.45563048 -0.35725474 -0.31684375 0.2157779; 0.50428843 0.3661609 -0.41359842 0.25081682 0.24480768 -0.17157346 0.41473097 0.32600915 0.39124355 0.21812738 -0.31839368 -0.19722098 0.45101544 -0.3368023 -0.20678613 0.43784016 0.4182857 -0.16936661 0.5184622 -0.26472545 -0.37700924 0.34229323 0.17902336 0.4332065 0.42007747 0.29268092 0.43482095 -0.080319256 0.2758412 0.25342444 -0.33323014 0.391513 0.426998 -0.41167146 0.1454227 0.37286934 0.4506067 0.38877133 0.36738217 -0.4247176 -0.19064261 -0.15321341 0.4261495 -0.49848565 0.21991801 -0.3759829 -0.29620352 0.12794213 0.34188128 -0.20515117 0.2886572 0.29831088 -0.13101803 -0.24301629 0.49221036 0.36408934 -0.29396567 -0.46310943 -0.08398891 0.3352558 -0.3848054 -0.2434245 -0.32146806 0.32541344 -0.45177856 -0.21749936 0.45598346 0.20172669 -0.30290553 -0.381055 -0.36787212 -0.36957192 0.20006949 0.18812653 0.30823076 -0.3597434 -0.122766614 0.2538483 0.44102854 -0.3545886; 0.41286814 0.56471837 -0.45606202 0.13270862 0.4895945 -0.3469689 0.0704104 0.3110209 0.3676122 0.3340433 -0.18841743 -0.389266 0.38308126 -0.24869816 -0.44552985 0.30529168 0.3323382 -0.19487597 0.25027308 -0.078505166 -0.46691766 0.55750823 0.34016988 0.13753782 0.32218456 0.23016328 0.47271174 -0.40179655 0.53439075 0.18980524 -0.2508669 -0.08157171 0.3012391 -0.24002531 0.3280103 -0.15642895 0.2945229 -0.078359134 0.1270662 -0.4773421 -0.21335907 -0.35291764 0.37162063 -0.08610934 0.32031018 -0.14934047 0.3347592 0.1269223 0.08020058 -0.26411286 -0.05213475 0.044259932 -0.28840056 -0.015264088 0.32809848 0.3849739 -0.18771945 -0.42345068 -0.17327066 0.2938334 -0.23971896 -0.1489822 -0.49869505 0.43220574 -0.036752306 -0.2807065 0.5354015 0.3929815 -0.16885558 -0.23995763 -0.20139341 -0.24958543 0.18750128 -0.020028874 0.09409243 -0.060047317 -0.24389037 0.30767155 0.1225798 -0.27447933; -0.2995228 -0.24944155 0.24394093 -0.16266666 -0.4075051 0.34028262 -0.2713141 -0.67006016 -0.4340376 -0.28641582 0.34899014 0.28809816 -0.51366615 0.40418684 0.18769123 -0.07206172 -0.34882292 0.24155396 -0.31909215 0.39773133 0.42453477 -0.46518862 -0.34743464 -0.2373704 -0.25548777 -0.27340728 -0.22874811 0.39788997 -0.4005245 -0.42814142 0.45921895 -0.2754917 -0.19208583 0.34738877 -0.14163458 -0.10779454 -0.18560565 0.094728865 -0.17376801 0.3726286 0.14409532 0.12936051 -0.48648533 0.33011904 -0.29484302 0.15539256 -0.004401691 -0.1561175 -0.08358795 0.46569502 -0.33538032 -0.1577494 0.46810576 0.39620152 -0.5072673 -0.35454202 0.20315082 0.28233466 0.007692592 -0.15128279 0.21100986 0.32220462 0.27406475 -0.26910004 0.28809536 0.33595228 -0.22713722 -0.25216374 0.46091226 0.46644226 0.17805155 0.49010512 -0.43571895 -0.39511827 -0.23575376 0.19394694 0.2999646 -0.421658 -0.33949092 0.46336675; -0.30139765 -0.3076868 0.17007601 -0.2436693 -0.41998878 0.3085587 -0.24735688 -0.10368919 -0.10789093 -0.37688214 0.23247808 0.15294687 -0.37345436 0.0847448 0.36270693 -0.10089675 -0.16496892 0.29247317 -0.49253172 0.17280942 0.3134222 -0.28216234 -0.3349359 -0.38602498 -0.44471356 -0.13494618 -0.2669304 0.023369173 -0.36031908 -0.5219472 0.32467017 -0.3056191 -0.41095424 0.52815026 -0.18170696 -0.35592324 -0.39267996 -0.36555392 -0.48464736 0.49343628 0.49222857 0.25703725 -0.37989613 0.45984304 -0.2940627 0.23189457 0.2077904 -0.20528178 -0.24721745 0.23109773 -0.09585542 -0.3071292 0.36312386 0.15323499 -0.5010498 -0.22690554 0.49537373 0.4347985 0.41067892 -0.25272834 0.16119702 0.24427074 0.32730728 -0.27644894 0.26377878 0.23462246 -0.45337114 -0.36718896 0.23486468 0.39394844 0.16355467 0.25190863 -0.49160707 -0.35271156 -0.19400446 0.1574894 0.069781415 -0.13486311 -0.1676051 0.37345654; 0.33718437 0.60053927 -0.5237189 0.3181099 0.17321849 -0.13033444 -0.56453127 0.5541662 0.023390155 0.19056717 -0.32575297 -0.5228064 0.6881126 0.29773146 -0.34929547 -0.1342007 0.4220237 -0.29209372 0.53917015 0.38878566 -0.36786047 0.84439844 0.18606187 -0.0859419 0.8403665 0.26694602 0.17254242 -0.58496386 0.5422397 0.34824085 -0.29794192 -0.25405627 0.67258173 -0.45117196 0.06942483 -0.30653545 -0.003691739 -0.08505474 -0.03720519 -0.52236575 -0.2150268 -0.118939884 0.45948562 0.07636328 0.04797342 -0.36051965 0.5201644 -0.37814644 -0.22930688 -0.32312495 -0.74428266 -0.4056262 -0.15586601 0.18506607 0.6865424 0.5754395 -0.19381544 -0.7015183 0.15839277 -0.23774149 0.34145677 -0.48856676 -0.5992925 0.11789792 0.2871737 0.05166734 0.09354309 0.14751768 -0.08661757 -0.47797716 -0.4176294 -0.75357187 0.52318907 -0.44080496 -0.016715989 -0.110964194 -0.21799843 0.22444357 -0.28213575 -0.5702856; 0.30212364 0.13317122 -0.12976845 0.1635241 0.37895826 -0.39576143 0.38880822 -0.55635065 0.5132899 0.19025825 -0.4840636 -0.12696156 0.25279492 -0.47131628 -0.24305968 0.37007958 0.3232775 -0.27828938 0.2844312 -0.37433445 -0.30700737 0.025153892 0.48799965 0.23104559 0.19077994 0.26492947 0.22996114 0.4961096 0.2846518 0.25304705 -0.09243327 0.2250981 0.1969743 -0.18009326 0.29705766 0.2643532 0.47852507 0.46820414 0.47876474 -0.14491455 -0.38442627 -0.37802476 0.3513481 -0.18240891 0.17759207 -0.30360818 -0.3387423 0.34770384 0.42793167 0.02878618 0.38402882 0.24126548 -0.334393 -0.4578075 0.050156187 0.32473326 -0.020798778 -0.22649 -0.3293539 0.26197666 -0.45719773 -0.20554516 -0.14848542 0.30424398 -0.17793372 -0.49252608 0.27083704 0.34094104 0.03857078 -0.2207265 -0.2316211 -0.07169475 0.116366215 0.30167082 0.33376586 -0.20097087 -0.20616372 0.40022495 0.4094521 -0.41013038; 0.050732214 0.3773537 -0.16476588 0.14343792 0.16556145 -0.038150594 -0.3740421 0.3831353 -0.061464135 0.28352383 -0.08296659 -0.30298388 0.2074298 0.14324255 0.039796103 -0.1814554 0.25985876 -0.20136961 0.40042934 -0.048727836 -0.082985274 0.424004 0.022971263 -0.34121433 0.38240656 0.08206984 0.13090435 -0.21690701 0.1619291 0.22240458 0.009760269 -0.10466376 0.40013078 -0.39207134 0.04075309 -0.34449333 0.21299425 -0.26266754 -0.11839765 -0.4914297 -0.14276165 -0.26258796 0.11569373 0.030790629 -0.12934066 -0.22954142 0.077572405 -0.029368948 -0.0988549 -0.267019 -0.2128152 0.13892323 0.07637085 0.2762758 0.18866724 0.19755976 0.04845153 -0.31161785 0.2632564 0.06785562 0.13060147 -0.21486172 -0.08841921 -0.16350064 -0.06387191 -0.17892648 0.317314 0.11810801 -0.17892638 -0.25287536 -0.09246912 -0.22416283 0.0781359 0.04756336 0.046412703 -0.18696775 -0.36074284 0.0083132405 -0.24535689 -0.28104833; 0.26028666 0.38515198 -0.36497447 0.24239708 0.37420768 -0.2938395 0.23296405 0.0617108 0.29834116 0.4275607 -0.11587302 -0.3681196 0.14370085 -0.17723405 -0.3750983 0.31097052 0.19012068 -0.41064292 0.39810097 -0.18698712 -0.113016374 0.26120505 0.37530464 0.16384402 0.22186433 0.26507583 0.22976792 -0.1671522 0.13022552 0.12435267 -0.08753683 0.35434797 0.38000837 -0.38608554 0.32914054 0.16032548 0.3441985 0.23313288 0.43724585 -0.41516706 -0.26765797 -0.2928277 0.2698828 -0.3025307 0.2437722 -0.34821358 -0.03336258 0.24668965 0.11940251 -0.28746715 0.155426 0.3605225 -0.37187913 -0.2546648 0.23586214 0.18316144 -0.27044043 -0.091890305 -0.40445703 0.4723662 -0.49931094 -0.16248737 -0.2957368 0.3650443 -0.31583762 -0.3782212 0.44300303 0.43486694 -0.23588528 -0.33864206 -0.38153142 -0.4454072 0.41282284 0.36931232 0.43229502 -0.20209874 -0.40825123 0.3835964 0.2976013 -0.3758162; -0.074150264 -0.05023501 0.014671362 0.35371718 0.0076658404 -0.45998615 0.5380132 -0.3997981 0.19106181 0.38909456 -0.42121664 -0.06669746 0.13424754 -0.41273242 -0.21190426 0.28864324 -0.018636877 -0.101384945 0.19647625 -0.51072323 -0.54275167 -0.13673906 0.18717842 0.1728827 0.058430765 0.41668296 -0.2888365 0.2654219 0.48061684 0.056254 0.1328355 0.21338 0.18310364 -0.20712213 0.25231603 0.29919323 0.4899885 0.22784074 0.5555215 -0.027125705 -0.23985721 -0.11232352 0.06678059 0.023767151 0.28552642 -0.16471212 -0.43520164 0.2506284 0.28672495 0.11863632 0.20374724 0.2995022 -0.26286522 -0.36525202 -0.17742163 0.28048226 -0.041762862 -0.024339013 -0.46876314 0.124401055 -0.13992651 -0.27924508 0.104250476 0.33091828 0.059017375 -0.4976431 0.13620476 0.4627983 -0.043879148 -0.3160293 -0.28584173 -0.30519745 -0.28784603 0.3393913 0.5671132 -0.16323958 -0.41956544 0.31380725 0.1624257 -0.20888159; -0.2734956 -0.27341878 0.08073289 -0.35796303 -0.34911546 0.43695366 -0.08692834 -0.56433034 -0.21172132 -0.37027085 0.37526512 0.14741023 -0.34674576 0.43450752 0.19349764 -0.13765441 -0.20783567 0.36630693 -0.23126644 0.33234352 0.13099146 -0.31269583 -0.11649276 -0.12219431 -0.37798557 -0.4860726 -0.0638573 0.21966928 -0.26267704 -0.41483653 0.19934286 -0.25177613 -0.38859552 0.10581241 -0.43897524 -0.3009663 -0.46656245 -0.23362255 -0.20133509 0.32112184 0.31279418 0.13365555 -0.3359218 0.35330743 -0.21845701 0.18864998 -0.21036193 -0.3354587 -0.312671 0.27343243 0.060363393 -0.23103526 0.44811162 0.41904193 -0.6240959 -0.3615043 -0.037265636 0.230928 -0.066563554 -0.1332694 0.04228896 0.20778522 0.23844595 -0.41229412 0.19474387 0.24547696 -0.14923167 -0.3786915 0.16259773 0.5728324 0.2913906 0.22593053 -0.112145945 -0.23282284 -0.20157081 0.027204229 0.5061395 -0.34464753 -0.07365697 0.54785234; 0.027288258 -0.25875485 0.3180897 -0.13702305 -0.21048704 0.20604655 -0.4261733 -0.09207294 -0.48698986 -0.35974467 0.47646037 0.33281973 -0.42317617 0.40993708 0.49908355 -0.36793968 -0.30960593 0.22822067 -0.338618 0.4337598 0.40121463 -0.41883716 -0.29818898 -0.30114093 -0.32499984 -0.23920879 -0.07567796 0.005781492 -0.23027356 -0.16836275 0.037512776 -0.34938872 -0.46231553 0.3912965 -0.50272197 -0.4757786 -0.571988 -0.20314297 -0.2710579 0.07427667 0.17982262 0.23556063 -0.46393645 0.05585206 -0.5141512 0.09482917 0.1294037 -0.47419545 -0.49165842 0.043978635 -0.28458628 -0.49333754 0.38021603 0.33311084 -0.2148906 -0.38952503 -0.010527752 0.33083555 0.20846431 -0.40919763 0.22937362 0.3294014 0.2492686 -0.4433324 0.22456837 0.29933235 -0.33686525 -0.38989067 0.18237239 0.45540038 0.19792798 0.1967159 -0.026382234 -0.32326746 -0.18104835 0.3621775 0.23223625 -0.3245864 -0.38387796 0.4181155; 0.3951212 0.15935306 -0.13822353 0.11160786 0.24350913 -0.3647038 0.19375245 0.6407633 0.12477354 0.46512777 -0.41770643 -0.43527162 0.1757499 -0.3532129 -0.42351028 0.2585268 0.47466373 -0.22909778 0.19043139 -0.39042142 -0.285793 0.26125374 0.42324075 0.40124282 0.31477892 0.40024054 0.41314924 -0.06442766 0.43772608 0.23095742 -0.061062366 0.382369 0.40815565 -0.30291274 0.30185068 0.2837675 0.25069624 0.3116834 0.35770047 -0.17512015 -0.29081944 -0.1915173 0.1278005 -0.19509573 0.28517324 -0.18251991 -0.2231756 0.46513042 0.4153983 -0.34802058 0.32890838 0.4519611 -0.24458665 -0.52072346 0.2226207 0.23934853 -0.12209212 -0.14662841 -0.13603304 0.27024406 -0.26743305 -0.4520656 -0.31654343 0.13172542 -0.27528036 -0.33169222 0.4642065 0.17151758 -0.12960257 -0.45999005 -0.13853247 -0.25845605 0.22944546 0.4338609 0.37832615 -0.45820543 -0.20564443 0.31412864 0.4364052 -0.44285947; 0.06352625 0.25005525 -0.41588223 0.27701455 0.23904644 -0.16023433 0.5194091 -0.1791781 0.3357894 0.16050515 -0.37469944 -0.2569522 -0.15547821 -0.5704507 -0.26442766 0.06535146 0.46989876 -0.28398812 0.013979259 -0.33175695 -0.114166975 0.06735314 0.3897509 0.27440393 0.37073314 0.39275432 0.29433003 0.013203051 0.3132078 0.03692319 -0.1268149 0.10484179 0.25067982 -0.29203197 0.41855967 0.27367753 0.09453858 0.2411952 0.3426005 -0.14305425 -0.24686436 -0.44192266 0.3193498 -0.24352647 0.47886693 -0.49449924 -0.29338977 0.5029974 0.32320735 -0.08817044 0.41177014 0.5080356 -0.47030646 -0.3625692 -0.10572255 0.16424732 -0.13584001 -0.032461643 -0.37056315 0.6230151 -0.3255188 -0.33683842 -0.07978718 0.2818861 -0.25212732 -0.36108246 0.19888854 0.3318475 -0.1906502 -0.26747894 -0.30897412 -0.32892996 0.12337525 0.35674164 0.3449734 -0.37075055 -0.33350068 0.38518348 0.41105166 -0.2189166; 0.35733446 0.48208278 -0.19663374 0.3495972 0.27303 -0.3031445 0.27735752 0.3537741 0.1588774 0.28560394 -0.3334551 -0.20657358 0.2697836 -0.41746214 -0.34857813 0.12172618 0.18380913 -0.25103787 0.51090735 -0.23194951 -0.47995982 0.5508278 0.3171748 0.16601303 0.3408241 0.48513213 0.4369031 -0.39594403 0.25255015 0.21022663 -0.52532446 0.07896641 0.19126719 -0.18705915 0.4276753 0.25295058 0.37421313 0.06257198 0.1998592 -0.14588882 -0.22186218 -0.3647722 0.28156412 -0.24247864 0.45161507 -0.14551385 -0.08743755 0.09852592 0.33866343 -0.45069265 0.19507548 0.30600417 -0.32508457 -0.10531271 0.4662488 0.28430572 -0.321811 -0.30216676 -0.052207362 0.16221248 -0.37939408 -0.1684531 -0.31805748 0.12367857 -0.28325278 -0.43817568 0.38531873 0.1691279 -0.54731125 -0.4798966 -0.20450155 -0.20354292 0.43929672 0.10672655 0.4765659 -0.22775269 -0.38848048 0.38507196 0.27171808 -0.19663739; 0.34851974 0.25373006 -0.33468226 0.33428437 0.34660017 -0.4915227 0.1545207 -0.018177932 0.38006267 0.49231732 -0.15174471 -0.34534806 0.3642621 -0.43097118 -0.21473046 0.46068168 0.4274856 -0.49679095 0.5374188 -0.37672296 -0.41152152 0.36039007 0.5632989 0.16051522 0.360356 0.4265292 0.37386274 0.13284478 0.34901428 0.29678366 -0.17789486 0.29480013 0.21762815 -0.5286333 0.34921998 0.13466525 0.31684825 0.4639937 0.23985784 -0.16523735 -0.24291915 -0.1996666 0.45770842 -0.31936675 0.29130158 -0.34382698 -0.100496426 0.19041172 0.596994 -0.39212278 0.49882632 0.378466 -0.26030734 -0.49529463 0.31163943 0.3845481 -0.28535065 -0.157349 -0.22061932 0.2972829 -0.5434851 -0.31753922 -0.2810472 0.34170008 -0.09276781 -0.49967143 0.21252301 0.19497591 -0.18310444 -0.4591834 -0.19376284 -0.46611026 0.121661365 0.44971332 0.44761616 -0.25749317 -0.23093928 0.515706 0.25191215 -0.5114193; -0.4301677 -0.30467412 0.46169916 -0.22336727 -0.36115283 -0.042609796 -0.18412988 -0.21801087 -0.086055495 -0.25448567 0.18197672 0.22093783 -0.23836997 0.19711004 0.021079117 -0.12522885 -0.32763532 0.17731144 -0.13637114 0.43925866 0.18637203 -0.30570146 -0.2748273 -0.011189907 -0.434892 -0.48894376 -0.3642351 0.2536284 -0.11754139 -0.5227422 0.525273 0.13851866 -0.067206606 0.44818172 -0.25778663 0.23330186 -0.2604158 0.24894124 -0.2628183 0.17707829 0.14844106 0.21842566 -0.23917474 0.40981486 -0.14932793 0.48693818 0.1219989 -0.11657051 0.12216035 0.4765111 -0.13744459 -0.13644406 0.14653815 0.20815249 -0.3651828 -0.17066729 0.18454841 0.20245333 -0.092090204 -0.22231688 0.14682412 0.20592152 0.2773492 -0.08794444 0.14896758 -0.028699646 -0.08832535 0.05513063 0.18407226 0.26565596 0.2901688 0.059389036 -0.37152448 -0.20895238 0.08791146 0.24308385 0.23178524 -0.4353159 -0.37055147 0.30481458; 0.280066 0.27469552 -0.15251955 0.41702685 0.22464114 -0.39247873 0.4774713 0.1629322 0.43333846 0.14429921 -0.43184847 -0.4362321 0.20620276 -0.22814012 -0.3698487 0.17713669 0.48576447 -0.28228405 0.39221004 -0.16795535 -0.18485904 0.44310725 0.34287927 0.06551253 0.15584704 0.27887926 0.5109647 -0.47833574 0.30775905 0.3920296 -0.23950745 0.21577302 0.14280081 -0.23446707 0.41100708 0.037440516 0.42677405 0.2540599 0.122872755 -0.19219431 -0.48863107 -0.26303282 0.16556653 -0.18637478 0.11651789 -0.20563754 -0.19074737 0.15694307 0.29447198 -0.1943278 0.16360009 0.17427401 -0.13162474 -0.14631934 0.17611961 0.20586638 -0.45025617 -0.4914698 -0.36304983 0.17352542 -0.17825723 -0.49861234 -0.41432023 0.15583424 -0.24886435 -0.39571598 0.3406945 0.24403852 -0.30312404 -0.46067962 -0.1329458 -0.36898082 0.38595772 0.4272842 0.3755269 -0.16661218 -0.45821756 0.13659869 0.31339082 -0.46981245; -0.15576254 -0.3739674 0.5603772 -0.3633678 -0.431565 0.31238788 -0.042097982 -0.6845736 -0.34657037 -0.35485807 0.1836217 0.38718608 -0.32975125 0.20946214 0.28087115 -0.32717457 -0.4158096 0.2804774 -0.17455997 0.09207489 0.3759675 -0.68762964 -0.22066808 -0.26874778 -0.22337484 -0.2290622 -0.26140258 0.2871886 -0.07776198 -0.45986792 0.4174402 -0.30862764 -0.2504941 0.35008365 -0.19320437 -0.015804524 -0.42195615 -0.12278326 -0.19984029 0.3915872 0.33482492 0.37278983 -0.43247095 0.1481791 -0.441933 0.47365332 -0.3398195 -0.09986347 -0.16676977 0.38587034 0.2289517 -0.074550964 0.30692166 0.17108805 -0.52928805 -0.3885665 0.26997277 0.47490564 0.13602138 -0.48667443 0.30440623 0.33251575 0.48526993 -0.43246338 -0.0009850196 0.2947576 -0.12766348 -0.24372375 0.20167796 0.18639491 0.23296297 0.47030753 -0.58842236 -0.21497186 -0.13955696 0.28012988 0.12052218 -0.30179903 -0.40626997 0.22629164; 0.47706407 0.3150439 -0.13527425 0.24516375 0.44881958 -0.3081614 -0.059451867 0.42736778 0.38120115 0.22510347 -0.47054335 -0.17440079 0.42625654 -0.121443756 -0.3124165 0.076914914 0.07743539 -0.14620237 0.46302086 -0.289247 -0.37515628 0.3636539 0.26498085 0.24622144 0.20795402 0.064359635 0.21461976 -0.14391807 0.38004512 0.28635573 -0.20726126 -0.24664278 0.26185533 -0.31739932 0.33597523 0.12628904 0.3722877 0.0027802526 0.1864519 -0.3788804 -0.1910886 -0.15730774 0.46660167 -0.13647771 0.124436654 -0.10960693 0.35946268 -0.06573026 0.07669983 -0.099542305 -0.08736498 0.1631064 -0.23832408 -0.11086759 0.6381343 0.28840408 -0.09227847 -0.34748706 0.12295018 0.32313108 -0.20994923 -0.21635745 -0.48309964 0.42949274 -0.008735466 -0.15583165 0.33362922 0.4132294 -0.12635861 -0.57272726 -0.11404796 -0.51064914 0.29791763 0.26106757 0.03917366 -0.006992189 -0.39509803 0.15081243 0.10549446 -0.28215227; -0.24665949 0.025123164 0.32252455 -0.31118327 -0.41477412 0.22512783 -0.25072634 0.34739003 -0.48209813 -0.22054271 0.18795656 0.19936903 -0.04970051 0.41167828 0.43999782 -0.33937117 -0.107914396 0.50647134 -0.034240685 0.37245196 0.4125328 -0.038489703 -0.24382515 -0.4577214 -0.1919179 -0.17460035 -0.20015167 -0.19829497 -0.1625774 -0.22265314 -0.012376704 -0.16124544 -0.017621245 0.34483388 -0.2241021 -0.18775803 -0.45645067 -0.23682389 -0.22618294 0.2755945 0.20311227 0.17598441 -0.20849551 0.3963426 -0.34574196 0.17120314 0.5312673 -0.4398496 -0.3832787 0.047085293 -0.49725136 -0.5348288 0.26056343 0.23729631 0.06120183 -0.34729528 0.20040654 -0.09220381 0.2594672 -0.16961153 0.22496794 0.35573432 0.09985923 -0.37636667 0.29532012 0.4976356 -0.36913544 -0.27045125 0.2030261 0.24372226 0.14810126 0.37323198 0.09243638 -0.3794219 -0.47815043 0.4953387 0.35142994 -0.22241691 -0.17364271 0.18567516; -0.16158624 -0.25553674 0.35181406 -0.33240956 -0.26695788 0.39508304 -0.1963794 -0.66086215 -0.19621104 -0.26051077 0.08809324 0.39937156 -0.097102284 0.09892819 0.32803297 -0.43581194 -0.51849556 0.14510134 -0.42037752 0.28931707 0.095411815 -0.35448185 -0.18382032 -0.12388081 -0.37378153 -0.16251943 -0.33350673 0.28639236 -0.28011495 -0.27573302 0.21856225 -0.12562557 -0.3294312 0.41191384 -0.19989683 -0.35738048 -0.40252578 -0.08692644 -0.38002983 0.13763793 0.5013216 0.46092665 -0.34842902 0.42709863 -0.16974863 0.45947653 0.23763953 -0.15416372 -0.47454688 0.5130331 -0.26091796 -0.14548227 0.541972 0.23182568 -0.4542746 -0.0056306347 0.455477 0.12539211 0.197082 -0.41375732 0.4962623 0.3100622 0.49558985 -0.041349214 0.48195165 0.1427568 -0.391566 -0.39957502 0.29441616 0.16621882 0.2625742 0.22340477 -0.4156288 -0.26604503 -0.17558135 0.44182834 0.16451685 -0.2570885 -0.32695743 0.10427812; -0.21140143 -0.00618085 0.19389196 -0.46036845 -0.4448225 0.41804624 -0.27538908 0.30335373 -0.15673977 -0.45921117 0.2722081 0.25438496 -0.15242453 0.3736887 0.37869892 -0.113261975 -0.28157502 0.41082054 -0.28411987 0.50406754 0.14380644 -0.07363725 -0.45099315 -0.34858665 -0.25274932 -0.1811592 -0.25519148 -0.20237705 -0.044788964 -0.09987388 0.29137406 -0.34879446 -0.09414727 0.20557034 -0.3484048 -0.23143251 -0.31376454 -0.14660646 -0.34275874 0.1163529 0.45849982 0.38039508 -0.21997239 0.37907115 -0.5049063 0.5420525 0.62576723 -0.46155673 -0.28527164 -0.012260758 -0.3574484 -0.49676505 0.27810898 0.28718016 0.12687078 -0.31848466 0.38663143 0.13884702 0.13336572 -0.2850932 0.38618442 0.29142308 0.19040363 -0.2772657 0.5062755 0.30451998 -0.15515138 -0.09911727 0.38539678 0.31060302 0.43272182 0.3631212 -0.40063977 -0.43932155 -0.3781606 0.31331944 0.41874072 -0.23182335 -0.31148112 0.37481976; -0.17620918 -0.5064399 0.010154139 -0.03419447 0.025076663 0.24765791 0.10095062 -0.27610216 -0.43907386 -0.41546983 0.61381346 0.5088127 -0.36606556 0.36574706 0.42469403 -0.1455274 -0.150946 0.22847733 -0.56412566 0.04560539 0.61823905 -0.26495665 -0.43498713 -0.3237615 -0.30723187 -0.2221557 -0.016364219 0.041855905 -0.6095328 -0.22119035 0.16373031 0.0067658704 -0.2565834 0.16632964 -0.25846097 -0.24329439 -0.32750717 -0.00654457 0.00586457 0.33597118 0.043170385 0.34138396 -0.55175745 0.06790098 -0.34446922 0.15135574 -0.4493541 0.056024127 0.033691414 0.28166488 0.08768858 -0.2410151 -0.048681937 0.037972998 -0.6395383 -0.73268974 -0.16069588 0.51946896 0.13819464 0.047384053 -0.0028770273 0.09259527 -0.034074835 -0.46473762 0.054365583 0.3208581 -0.4089459 -0.2993998 0.11610867 0.61092687 0.08983022 0.3099716 -0.008810669 -0.11117309 -0.33287537 -0.021113403 0.7136579 -0.06308691 0.11037589 0.67936116; 0.14786074 -0.12358399 0.3615602 -0.35560587 -0.33380637 0.40004426 -0.4722856 0.22213565 -0.2264428 -0.2368273 0.29963997 0.37110656 -0.17089759 0.3652277 0.31879854 -0.24342841 -0.3940586 0.31733206 -0.22518988 0.49063826 0.40446526 0.10261467 -0.5508462 -0.1300138 -0.33661214 -0.45907104 -0.20557882 -0.08172983 -0.37558755 -0.1039581 0.16522335 -0.5070443 -0.4078222 0.14935225 -0.33683407 -0.35952672 -0.10113883 -0.26567894 -0.4159873 0.4279035 0.22354458 0.38847283 -0.02390286 0.30870387 -0.53616923 0.027009236 0.08816631 -0.22617887 -0.3309522 0.059450887 -0.5153425 -0.5463389 0.1484828 0.33483633 -0.093449 -0.16585906 0.057277497 -0.027132293 0.17763121 -0.45677084 0.25562558 0.35850656 0.28369823 -0.3530931 0.2762742 0.1724512 -0.44124627 -0.17027996 0.15539227 0.010923175 0.40502623 0.37043172 -0.1278091 -0.31158054 -0.22700967 0.24848586 0.3967781 -0.4329784 -0.15208945 0.23703264; 0.17198345 0.26565698 -0.22352822 0.15408699 0.21615839 0.090205796 0.0072874785 0.3107292 -0.23267917 0.095558874 -0.015883809 -0.37321678 0.2866357 0.02278455 -0.13335383 0.016148888 0.3927645 -0.43580633 0.111926645 0.0700219 -0.10097537 0.2572522 -0.26108953 -0.43835995 0.37397403 0.26933366 0.25057432 -0.41057178 0.01785432 0.24974643 -0.45124525 -0.24598065 0.3061432 -0.30276167 0.24986309 -0.29177678 0.08414611 -0.15593056 0.17440483 -0.45664725 -0.31678572 -0.05603124 0.04590428 -0.18328671 -0.008376391 -0.19546528 0.32052138 -0.00652925 -0.20314702 -0.13602227 -0.29651695 0.07852622 -0.46139842 0.15838109 0.07676961 0.22350802 -0.41284868 -0.3306199 0.30978483 0.10304824 -0.21196818 -0.3799036 -0.44313917 -0.009462132 -0.34190014 0.2786651 0.017894065 -0.030663539 -0.31900135 -0.05128695 -0.38300738 -0.3041322 0.3048669 0.07841313 -0.022714969 -0.3476384 0.065389894 0.18110709 -0.07717337 -0.018621393; -0.068316884 -0.23598123 0.08825573 -0.41857544 -0.3968527 0.30381313 -0.22194584 -0.0020473439 -0.32552317 -0.4342792 0.29934362 0.2918674 -0.32557818 0.3675423 0.34748384 -0.16485673 -0.21669973 0.08602513 -0.37166995 0.36902234 0.4370328 -0.2693978 -0.4811951 -0.4871448 -0.43546265 -0.17707121 -0.020020686 -0.32909736 -0.57047695 -0.008576795 0.20571338 -0.20261955 -0.18096465 0.06537949 -0.15899386 -0.452821 -0.18869232 -0.35462052 -0.40650445 0.36212856 0.2965127 0.14010753 -0.3929745 0.03702268 -0.225823 0.15461305 0.10239673 -0.14194025 -0.11737857 0.035307467 -0.43586367 -0.43690938 0.06477702 0.2955763 -0.31316787 -0.5685762 -0.10734756 0.4638738 0.268241 -0.3588403 0.15398124 0.19947964 0.084322326 -0.34453717 0.15460604 0.45783985 -0.4569574 -0.24649498 0.16339228 0.28226954 0.14800006 0.29272607 0.06033192 -0.22586761 -0.23445633 0.23692295 0.45668882 -0.15803315 -0.27761057 0.33527115; -0.50577146 -0.32709968 0.19247296 -0.38685402 -0.3293252 0.28948045 -0.107096575 -0.29292253 -0.34400436 -0.19885321 0.12164016 0.17964749 -0.3711998 0.2578312 0.18198878 -0.24004509 -0.43328315 0.37443405 -0.34984323 0.22905363 0.40296888 -0.19211477 -0.14237297 -0.34433487 -0.39918536 -0.27450752 -0.48597318 0.3103306 -0.28946677 -0.47690985 0.30004895 -0.11995878 -0.4735193 0.22965825 -0.15916663 -0.36394382 -0.099769495 -0.36039373 -0.28708816 0.2538452 0.18594357 0.4318322 -0.3939925 0.25473508 -0.10615826 0.1767486 0.15395229 -0.145176 -0.17695726 0.24745016 -0.15616307 -0.28821105 0.2682289 0.39623392 -0.31117186 -0.47175384 0.16782172 0.38459626 0.25921032 -0.23055728 0.25799316 0.47608578 0.37963927 -0.40731105 0.3318178 0.14040416 -0.45596308 -0.32138523 0.51701105 0.33548227 0.13750353 0.48280033 -0.24641451 -0.48187456 -0.43667948 0.11300529 0.3157056 -0.108135596 -0.19006665 0.33356944; 0.34646535 0.3317613 -0.44158617 0.25889972 0.37556335 -0.4134171 0.14055347 0.4356846 0.22592488 0.4011802 -0.19114746 -0.5873227 0.7025879 -0.18271844 -0.37912837 0.10097396 0.20989986 -0.49076676 0.5069472 -0.37584218 -0.31491145 0.35179397 0.07435021 0.049080417 0.31767884 0.15040176 0.38007343 -0.22519812 0.14391577 0.350863 -0.16135556 0.07991003 0.29192206 -0.4336221 0.2193169 -0.18835561 0.37455904 0.027918855 0.42828536 -0.24364163 -0.4528138 -0.4937647 0.15312836 -0.1663552 0.34351233 -0.30570182 0.5708039 0.34365064 0.08720624 -0.318172 -0.32461077 0.0620293 -0.21689117 -0.4119941 0.6905807 0.37657306 -0.003497313 -0.4990284 -0.07954017 0.4513676 -0.13799459 -0.5282494 -0.43415585 0.042523824 -0.011098492 0.031303715 0.4482019 0.16922669 -0.10368441 -0.26461005 -0.25412336 -0.20660356 0.39986718 0.16728383 0.08306574 -0.16656558 -0.40255094 0.4017387 0.07320663 -0.2958283; -0.19269499 -0.32848534 0.23155898 -0.45035726 -0.35780188 0.107498646 -0.42239624 0.4864152 -0.27326563 -0.18839486 0.33620697 0.12388988 -0.30132437 0.1918673 0.45039505 -0.4379277 -0.34878245 0.28586185 -0.077308774 0.21996233 0.3170572 -0.24742466 -0.15257926 -0.121262915 -0.23106545 -0.23744911 -0.29757854 0.0006977457 -0.29356363 -0.3452971 0.3292335 -0.1232442 -0.33769795 0.13155974 -0.360943 -0.27869347 -0.11015304 -0.29293686 -0.44022283 0.113060184 0.49787658 0.21776567 -0.09054756 0.1745461 -0.4479195 0.20666148 0.45545822 -0.5141162 -0.22726166 0.04895651 -0.30333906 -0.4531729 0.15241632 0.329492 -0.061086934 -0.19761048 0.41449833 0.34027457 0.30437702 -0.28190368 0.3250846 0.3921751 0.33124247 -0.19343379 0.4348853 0.35876143 -0.39622474 -0.40984932 0.36199334 0.124244876 0.44440165 0.44807947 -0.16536863 -0.3746479 -0.3956908 0.33926675 0.28015617 -0.25559825 -0.23253846 0.2613358; 0.15844214 0.37216064 -0.27745754 0.31228757 0.20048922 -0.21565606 0.3173977 -0.39353296 0.168348 0.31021264 -0.112627275 -0.112127006 0.2886787 -0.28604046 -0.33143196 0.3682942 0.38540596 -0.42175597 0.10395376 -0.29388413 -0.09918897 0.34292892 0.477284 0.06035753 0.2800993 0.29104316 0.29043576 0.0006595391 0.26836774 0.3613986 -0.43649405 0.16167651 0.43947467 -0.14738397 0.28654748 0.39627004 0.4295658 0.30959472 0.45599645 -0.109726384 -0.18985546 -0.29094583 0.45033237 -0.19899882 0.31130904 -0.47874615 -0.38648537 0.27730462 0.34681252 -0.11082623 0.39290142 0.18011086 -0.08957445 -0.22700106 0.17126189 0.083718166 -0.4368275 -0.32932138 -0.17588289 0.40408653 -0.35933584 -0.3606141 -0.32091215 0.13379183 -0.34044024 -0.17779621 0.13697347 0.36284077 -0.13732874 -0.25577903 -0.12899566 -0.26411134 0.1484294 0.451459 0.26653746 -0.23072705 -0.16839738 0.19577362 0.21505022 -0.10081034; 0.3355749 0.14730932 -0.39578477 0.2489695 0.3667305 -0.23052306 0.292166 0.3895144 0.35249028 0.49163097 -0.2663665 -0.30837062 0.4377039 -0.20616043 -0.36657476 0.19351386 0.21918097 -0.19408686 0.45990714 -0.25484142 -0.4410639 0.32909924 0.27985528 0.52581984 0.43731257 0.5038905 0.027110634 0.16819388 0.43180496 0.17507826 -0.08521231 0.18798737 0.42132562 -0.3606011 0.15565228 0.4151928 0.5694943 0.41181394 0.27358615 -0.35504854 -0.31606528 -0.31127223 0.39912567 -0.21934052 0.24455553 -0.05455436 -0.124423824 0.33607346 0.42396212 -0.18806918 0.30519968 0.26205426 -0.45017597 -0.4540844 0.39121807 0.30004874 0.044041198 -0.52248883 -0.4740716 0.3367442 -0.15328304 -0.31749433 -0.2960683 0.18493621 -0.25976622 -0.50963295 0.43915984 0.5282969 0.04185171 -0.25186914 -0.27559862 -0.21496208 0.0040029837 0.2834179 0.2972101 -0.11670494 -0.2522445 0.25193036 0.30363148 -0.40533096; -0.49959147 -0.2950234 0.30513725 -0.4187586 -0.43680432 -0.14643604 0.04598926 -0.50083727 0.24684192 -0.13069242 -0.035133764 0.24130487 -0.1557343 -0.08616426 -0.06856145 -0.051138088 -0.43064478 0.12539497 -0.13848206 0.20375662 0.27443472 -0.16641471 -0.08986847 0.30399194 -0.4500905 -0.31055683 -0.31298104 0.49245518 0.012340929 -0.31903267 0.45563042 0.2863271 -0.2547936 0.20803082 -0.31314707 0.07115841 -0.14663051 0.42611614 0.08020204 0.24703898 0.21960658 0.19642109 -0.10493825 0.09475082 -0.2127509 0.34947345 -0.09126035 -0.11138938 0.25854397 0.24799477 0.1881653 -0.091205455 0.46742144 0.28095058 -0.223275 -0.098554164 0.32682544 0.3083331 -0.28358456 -0.07479781 0.14637457 0.5164479 0.26653206 -0.09087207 0.090101056 -0.0842818 0.019219534 -0.09184431 0.2906145 0.2996522 0.36259204 0.28721535 -0.40907064 0.051134713 -0.093429804 0.24988464 0.2605863 -0.09028339 -0.15232214 0.07187305; -0.49029192 -0.4851259 0.50610644 -0.36843905 -0.2622503 0.23918332 -0.06288544 -0.7643753 -0.28888366 -0.27521202 0.27404523 0.5744276 -0.5844015 0.1652426 0.15976267 -0.19202478 -0.20158596 0.24669322 -0.38231066 0.064859726 0.13689373 -0.5306477 -0.1086971 -0.09434402 -0.3619461 -0.14832836 -0.47103944 0.3707402 -0.3036118 -0.4228851 0.32480457 -0.09164551 -0.29970166 0.516768 -0.39909044 -0.13460903 -0.35803398 -0.12656699 -0.028627949 0.6024041 0.32913303 0.18396471 -0.14882925 0.40917984 -0.10297736 0.26042694 -0.27227762 0.084033415 -0.09153461 0.28708684 0.15166883 -0.10292213 0.38281533 0.13520266 -0.54631805 -0.056673616 0.08937598 0.50026566 0.1472129 -0.21624939 0.25768918 0.44880772 0.40331882 -0.03966885 -0.06576697 0.07807432 -0.16618411 -0.41607815 0.32214683 0.27067238 0.45154947 0.3480421 -0.5375492 0.13016264 -0.20244539 0.13623331 0.13661817 -0.41293263 -0.21778107 0.358661; 0.028325196 0.19318819 -0.28867528 0.21692283 0.30736205 -0.14749064 0.5701332 -0.27779633 0.13974978 0.37329024 -0.11772906 -0.13129832 -0.21782039 -0.2521683 -0.17951421 0.31993324 0.329141 -0.2473117 0.008043768 -0.25404853 -0.16170438 0.12962614 0.51893 0.09616024 0.15429446 0.12077982 0.16035898 0.120413855 0.23484674 0.26213273 -0.24444799 0.42481345 0.35892415 -0.4443549 0.20874391 0.17766806 0.2940292 0.20940293 0.23127866 -0.26771146 -0.26509416 -0.43926594 0.11964651 -0.37727946 0.39500523 -0.4184935 -0.5430703 0.37809178 0.41737878 -0.05207006 0.57836163 0.13141862 -0.19158375 -0.46991953 0.046651855 0.37141946 -0.27141243 -0.0901129 -0.24519016 0.1812635 -0.25084758 -0.157307 -0.39342615 0.31385273 -0.44389856 -0.5006339 0.11797409 0.24801625 -0.25841218 -0.26478347 -0.39666873 -0.17885773 -0.074776694 0.40387362 0.47490653 -0.17727458 -0.37288356 0.09639577 0.45654094 -0.09816673; 0.23532756 0.4400112 -0.38191056 0.43355656 0.27426976 -0.26759532 0.06838146 0.35617676 0.109737925 0.32941312 -0.43050453 -0.3955716 0.41396847 -0.03877732 -0.18473415 0.4516669 0.12810774 -0.20746851 0.5301663 -0.32509497 -0.43543428 0.3562967 0.17609519 0.3780243 0.3908607 0.16180651 0.2344173 -0.49756375 0.44279662 0.19302875 -0.31369975 -0.009734169 0.555195 -0.30299208 0.13190523 0.058827855 0.44535682 0.15237345 0.17671792 -0.5089732 -0.22948618 -0.3178331 0.47901344 -0.11899236 0.13581975 -0.22938816 0.032052036 0.1578581 0.09561705 -0.5000487 0.2241028 0.09005227 -0.39821318 -0.21108632 0.25797763 0.35953194 -0.28548983 -0.34848657 -0.25213835 0.07798527 -0.17383797 -0.2750207 -0.3444226 0.46582422 -0.20120746 -0.19076088 0.27833736 0.44042298 -0.43720275 -0.15208443 -0.23279613 -0.4346384 0.28056842 0.14567934 0.42861608 -0.26568362 -0.29266298 0.1936287 0.30376586 -0.3829367; 0.21544608 0.4257745 -0.4889336 0.2838701 0.3625077 -0.43353593 0.33460286 -0.06293755 0.20546688 0.37704113 -0.48308584 -0.3646949 0.3427522 -0.38074133 -0.50374186 0.39287126 0.28605616 -0.2707142 0.24934609 -0.44988194 -0.3924097 0.38179097 0.41654617 0.3462878 0.18285298 0.19131036 0.3976903 0.1996656 0.43127978 0.24553566 -0.48323283 0.28978145 0.19432147 -0.16687877 0.17793323 0.31547394 0.40231633 0.44457138 0.2424108 -0.3190811 -0.33575237 -0.12206057 0.2636877 -0.4406209 0.3204576 -0.4132534 -0.41789272 0.11952309 0.17508492 -0.411143 0.45879754 0.11723033 -0.22137053 -0.13436419 0.39540735 0.35726243 -0.30974513 -0.32707065 -0.4220214 0.26245245 -0.3647363 -0.32788485 -0.41018832 0.5308091 -0.34430245 -0.1542792 0.4207111 0.38755676 -0.4977036 -0.16111143 -0.17591198 -0.47854042 0.18910812 0.21241586 0.3439271 -0.22228383 -0.13005075 0.13286555 0.23743853 -0.48105374; 0.16079685 0.2198667 -0.2513883 0.2550238 0.009952144 -0.53057086 0.17715807 0.63045037 0.39703143 0.22732346 -0.54538316 -0.4884178 0.3233826 -0.40419877 -0.16050373 0.13577642 0.13414119 -0.24506162 0.4312606 -0.30290005 -0.3482191 0.29342195 0.112564735 0.08939347 0.44189626 0.24494955 0.3656 -0.33849183 0.38093036 0.3645993 -0.04826825 0.04538846 0.34117463 -0.3288208 0.25284737 0.2435497 0.3578627 0.030281266 0.014485282 -0.29836938 -0.10613895 -0.32145333 0.29780406 -0.25356436 0.03972177 -0.06405434 0.12479239 0.28887713 0.22566222 -0.28531727 0.052506838 0.045710597 -0.36346388 -0.33589622 0.5211028 0.55119413 0.101473376 -0.462581 -0.22066107 0.19706878 -0.080750264 -0.2636836 -0.07468455 0.41625145 -0.20333225 -0.22077782 0.341997 0.53835386 -0.045809593 -0.5295825 -0.07815676 -0.14878114 0.24090786 -0.13318786 0.49553326 -0.13134883 -0.53644073 0.27367577 0.3026722 -0.49933836; -0.33408433 -0.36394036 0.5295057 -0.2250108 -0.19288653 0.39436004 0.39732006 -0.62746197 -0.30590746 -0.36823758 0.22006463 0.41691208 -0.71150434 -0.0691921 0.4582046 -0.14213641 -0.29769668 0.40158275 -0.45620343 0.24106933 0.38937253 -0.58406854 -0.1729448 0.03619702 -0.35294095 -0.29379606 -0.34704596 0.37508687 -0.1550118 -0.14954452 0.3156699 0.039600067 -0.4957551 0.43642297 -0.38027042 0.032370135 -0.102276035 0.20797543 -0.13025963 0.40211198 0.24982207 0.22547485 -0.37646624 0.20107614 -0.062241063 0.1741635 -0.56725615 -0.03705718 0.023361325 0.15773444 0.54396284 -0.092525735 0.109219514 0.016614309 -0.6191252 -0.1586193 0.07043936 0.5185446 0.15471391 -0.023217918 0.2550529 0.41219932 0.506199 -0.36081868 -0.060612068 0.26565984 -0.47093934 -0.22218688 0.06360452 0.49434143 0.27980912 0.56416374 -0.2984373 0.07139415 -0.28328717 -0.09350417 0.4686892 -0.34964925 0.14021838 0.4957434; -0.36333996 -0.4455159 0.45270726 -0.36455843 -0.41527462 0.3277524 -0.12869258 -0.7759376 -0.15363064 -0.33139238 0.21200353 0.42729685 -0.553222 0.1338191 0.17749453 -0.085968465 -0.27076963 0.12192063 -0.41412187 0.22133237 0.19361354 -0.5252792 -0.35098216 -0.088123724 -0.56805396 -0.28969088 -0.5392987 0.45135725 -0.24483143 -0.3174363 0.11612955 -0.021081481 -0.38340202 0.41112548 -0.40868026 -0.09392548 -0.39292592 -0.06600552 -0.10049857 0.36562243 0.30887878 0.18948679 -0.39902923 0.25488532 -0.13634421 0.5076292 -0.3477306 -0.32357866 -0.23438685 0.27256998 0.27325457 -0.18122444 0.42729893 0.21168326 -0.45247316 -0.22451398 0.08250379 0.2263431 -0.04870201 -0.4362755 0.2547554 0.38789216 0.3413486 -0.40127558 0.2722674 0.089402415 -0.42523178 -0.3992259 0.28130576 0.32369602 0.2785994 0.35851216 -0.5877071 -0.21228863 -0.35306668 0.3944454 0.17594826 -0.22267117 -0.37873918 0.45875925; -0.099073894 -0.39239088 0.19029653 0.06428806 -0.17749155 -0.18736774 0.30102557 -0.48968586 0.14747405 0.035634737 0.22295502 0.39260128 -0.40739945 -0.35032582 -0.024073197 0.12913796 -0.19504945 0.05576486 -0.3609549 -0.17731635 0.033398446 -0.6260468 0.18488742 0.19319859 -0.68543464 -0.26966313 -0.22861098 0.5071107 -0.032033935 -0.37074083 0.059338797 0.32458124 -0.30945513 0.39717355 0.18467975 0.15828562 0.005584111 0.16589311 0.39282408 0.64788485 0.047774546 0.053411856 -0.39848572 0.11171045 0.33814755 0.105110034 -0.47934493 0.24609321 0.35786757 0.31853667 0.56232035 0.20484397 0.15909585 -0.2971978 -0.6298321 -0.087480165 -0.09857301 0.4724408 -0.080399625 0.32380787 -0.17734806 0.20385769 0.34716666 -0.079096444 -0.025406245 -0.05366162 -0.20669931 -0.15884979 0.17928813 0.3367646 -0.064484335 0.40836167 -0.1967568 0.14810379 0.19840014 -0.10438865 0.06800608 -0.22829771 0.18595038 0.4552841; -0.22999536 -0.3766996 0.017317474 -0.26525584 -0.38236895 0.2646332 -0.23697656 0.01138158 -0.5083553 -0.32588327 0.5257816 0.42047247 -0.10442077 0.24750523 0.2504875 -0.42744866 -0.30965245 0.14474064 -0.45295805 0.2345013 0.3517981 -0.16105942 -0.3732716 -0.24909325 -0.34778598 -0.48517236 -0.19968273 -0.34665698 -0.47798422 0.008772284 -0.015067966 -0.22063486 -0.50116485 0.44149444 -0.37537223 -0.19819058 -0.6209892 -0.37662885 -0.26642525 0.281469 0.4170957 0.28650612 -0.4130215 0.31708464 -0.18947953 0.18727918 0.11248987 -0.36562085 -0.19845535 0.20469202 -0.39266092 -0.30571637 0.40533537 0.4015724 -0.35809857 -0.43518397 0.1952042 0.5324941 0.36918926 -0.15476783 0.2877916 0.19735101 0.18738374 -0.51875675 0.07996074 0.48662376 -0.40155438 -0.23207894 0.23631936 0.27820888 0.42265537 0.18200716 -0.26263225 -0.24332449 -0.43857077 0.06546329 0.5402227 -0.41561455 -0.03327164 0.53229415], bias = Float32[-0.16958217; -0.18340582; 0.14819516; -0.2619049; -0.30644464; 0.29619706; -0.19585596; -0.6117904; 0.24024361; 0.24509004; -0.654741; -0.3164988; 0.2877824; 0.2365119; -0.20314436; 0.23680842; 0.4135898; 0.29956692; 0.07060499; -0.6808732; 0.19327918; 0.2545472; 0.09712252; -0.23536825; 0.020152595; -0.32841453; 0.3060109; 0.27679136; 0.14409906; -0.014272958; -0.42606932; 0.42427918; 0.16247874; -0.4591237; 0.5772837; 0.21654236; -0.20184231; 0.24199888; 0.26408657; -0.21726117; -0.28850034; 0.22766773; 0.20211464; 0.054169092; -0.26547053; -0.26791808; -0.19111045; -0.39434764; 0.40032622; 0.35069165; -0.24589385; -0.12239555; -0.26101112; -0.088691615; 0.18762964; -0.2267282; 0.10196054; -0.3889095; 0.30107886; 0.2051446; 0.15316375; 0.71920747; 0.002746298; -0.068789594; 0.44536394; 0.2381113; -0.15137134; 0.17400855; -0.29579508; -0.40570134; 0.17327878; 0.12759146; -0.33335474; -0.27824602; -0.2353174; -0.46598876; 0.11818533; 0.09267592; 0.11569809; 0.45815003]), layer_3 = (weight = Float32[-0.4344142 -0.23854584 -0.2148162 -0.32726565 -0.037837747 0.17041562 0.55573785 -0.2918544 -0.078891605 -0.14860578 -0.62294465 0.47102302 -0.22532041 0.047906462 0.021097384 0.47271085 -0.22722968 0.06860179 -0.94541216 -0.41064456 -0.23654683 -0.34251246 0.8957223 0.075523555 0.48850393 -0.8255348 -0.06415381 0.49478203 -1.5252111 1.6050903 -0.57006407 -0.3018281 -0.5414825 0.2740998 0.2667954 0.6686849 -0.1830787 -0.22175017 -0.11929899 0.12364863 -0.26647672 -0.3201868 0.08336315 -0.858663 -0.6257153 0.5410642 0.6526265 0.21869798 -0.21368709 -0.37417153 0.5360023 0.023442937 0.20973648 0.6851965 0.49642032 -0.02294344 -0.80942947 -0.5511729 0.0782906 0.23153314 -0.023307271 0.478356 -0.4643946 -0.6219898 0.3846851 -0.05037751 0.30255082 0.0010303268 -0.42630747 0.46492946 0.8804969 -0.9699932 -0.42595 -0.25290027 0.025197828 0.28908336 0.8831079 -0.8518969 1.5539908 -0.4681123; -0.1389978 0.7394719 -0.6929102 0.08507313 0.2839259 0.04099713 -0.22003476 -0.0055367807 0.3558481 0.0065307803 0.594333 -0.4375747 0.4117595 -0.8319338 -0.22569609 -0.10172533 0.47965005 0.4531871 -0.32321474 0.05694364 0.37794647 0.5006184 -0.26891154 -0.34852704 -0.14885478 1.3268313 0.43004474 0.053524677 1.5046364 -0.12941088 0.31307158 -0.060339082 -0.26469377 0.13687757 0.8574386 0.04024273 0.37124956 -0.48084947 0.39385456 -0.19112866 -0.95789325 0.44410074 0.4395129 0.77038795 -0.106633626 -0.25881556 -0.1885484 0.37640876 -0.0029812385 0.10725708 -0.01879513 0.35473767 -0.32758337 0.5197857 0.14181353 -0.2687886 -0.09737784 0.18527065 0.32720318 -0.9987671 -0.30541027 -0.4482365 0.008598135 -0.011569706 -0.06089829 0.46146828 0.4266888 0.5704916 -0.2911672 0.0665517 -0.107578136 0.036791194 -0.03507208 -0.5228806 -0.28443295 0.8742884 -0.77426654 -0.08239694 -0.07625797 -0.06037654], bias = Float32[-0.85371286; -0.015588574]))


loss_neuralode(p_trained_moderate)





#---------------------------------------High noise collection -------------------

rng = Random.default_rng()
Random.seed!(99)

#Adding moderate noise to data:

x1=Array(sol)

x1_mean = mean(x1, dims = 2)
noise_magnitude = 35e-2
x1_noise = x1 .+ (noise_magnitude*x1) .* randn(eltype(x1), size(x1))
#-------------------------------------Defining the Neural ODE------------------------------------


dudt2 = Lux.Chain(Lux.Dense(2, 80, tanh),Lux.Dense(80, 80, tanh), Lux.Dense(80, 2))
#Setting up the NN parameters randomly using the rng instance
p, st = Lux.setup(rng, dudt2)


prob_neuralode = NeuralODE(dudt2, etaspan, Tsit5(); saveat = etasteps)

function predict_neuralode(p)
    Array(prob_neuralode(I, p, st)[1])
end


#training data
true_data= x1_noise[:,1:end-60]
### Define loss function as the difference between actual ground truth data and Neural ODE prediction
function loss_neuralode(p)
    pred = predict_neuralode(p)
    loss = sum(abs2, true_data .- pred)
    return loss, pred
end


p_trained_high = (layer_1 = (weight = Float32[-0.6100974 -1.5032985; -1.3287636 -1.8379558; 0.9023488 1.4983381; -0.88419574 -1.1436325; -0.90894395 -1.1673802; 1.5834048 2.1787963; -0.867481 -0.39458522; -1.119264 -2.821439; -1.4728149 -2.2910075; -0.9704229 -1.2734823; 1.564777 2.4518886; 1.3706592 2.1542528; -1.3014821 -2.3821487; 0.9205628 0.82016784; 1.4811652 2.064578; -1.5513664 -1.884433; -0.82133573 -1.2541732; 0.82791334 1.2008581; -1.3599741 -2.2388034; 0.89471275 0.79801196; 1.4686854 2.8934753; -1.2720969 -2.4224014; -1.5562562 -1.9020194; -1.6323501 -1.7210064; -1.356214 -1.939069; -0.9268746 -1.4584512; -0.613578 -1.655958; 0.53887707 1.7444429; -1.5565811 -3.2666857; -0.620596 -1.6607002; 0.46737576 1.0912427; -1.5266061 -1.6668051; -1.3312963 -2.5106306; 0.9284526 1.3464569; -0.8209819 -0.9689203; -1.6277734 -1.9227997; -1.6955593 -2.105331; -1.665728 -1.4447972; -0.88525623 -0.72211623; 1.2701422 2.8097332; -1.439382 -1.1518419; 0.94017243 1.1203374; -1.4950709 -3.0131538; 0.73559815 0.8082584; -0.87013113 -0.8322378; 0.63991475 1.100212; 0.93273705 0.26175022; -0.66229624 -0.37352473; -1.5721673 -1.4034843; 0.5622462 1.6133446; -0.9034504 -0.19663349; -0.7934334 -0.6624576; 0.72197765 1.1279032; 0.8888105 0.77540547; -1.3698797 -2.7564285; -1.5870705 -3.111938; 0.6753185 0.606154; 1.2501996 2.8696477; 1.5090969 1.3830813; -0.8686967 -0.8149204; 0.9389497 0.9632682; 0.8344183 1.2628176; 0.6920543 1.5749943; -1.471353 -2.025423; 0.78482306 0.6258931; 1.5989001 2.0944636; -1.5536878 -2.5422604; -1.4888166 -2.3191695; 0.6468106 1.025094; 1.5137143 2.7024932; 0.85158914 1.2773465; 1.3641049 2.7978444; -0.57905084 -1.6477195; -0.8931315 -0.60785306; -1.6582808 -2.1057925; 0.71683997 0.8767851; 1.6673367 3.2819288; -0.882331 -1.2489604; -0.8614499 -0.9680857; 1.5241193 2.6747913], bias = Float32[0.08468323; -1.1233885; 0.22670357; -0.18925926; -0.26467642; 1.1859136; -0.20154767; -0.99524736; -1.2331419; -0.29589683; 1.27699; 1.119608; -1.1559258; 0.20943229; 1.1629741; -1.4916291; -0.07153246; 0.15522419; -1.1355691; 0.18049204; 1.0098362; -1.0896873; -1.0753938; -1.4355117; -1.1561992; -0.17202559; 0.11708564; 0.78303707; -0.9847026; 0.097608; -0.24401978; -1.2425138; -1.0737325; 0.8021107; -0.11493314; -1.421173; -1.5583684; -1.4307939; -0.17079538; 1.1715926; 0.27409294; 0.24884608; -1.1894091; 0.057588935; -0.113908894; -0.07122052; 0.30790022; -0.06547959; -1.3589319; 0.036087144; -0.26094192; -0.06394986; 0.022300653; 0.22089097; -1.3020611; -1.1229291; 0.2251894; 1.1288419; 1.2934686; -0.23858847; 0.606002; 0.16335028; 0.12286934; -1.2314663; 0.23627736; 1.3911476; -1.2561685; -1.3498583; 0.09527712; 1.3332839; 0.17702821; 1.0001994; -0.07739874; -0.36349288; -1.3966541; 0.20301886; 1.0858953; -0.18238553; -0.40632486; 1.2286191]), layer_2 = (weight = Float32[1.4753685 -0.5911841 -1.6355942 1.4221979 1.5095464 0.7985545 1.1806843 -0.704434 -0.6071017 1.6717093 0.5777676 0.5747645 -0.7436027 -1.4183868 0.6998803 -0.6046477 1.4229515 -1.4784093 -0.6064929 -1.4688357 0.6113861 -0.7239163 -0.67643535 -0.6004645 -0.8282175 1.5755345 1.4144069 -1.4188807 -0.37490436 1.456243 -1.2416364 -0.5753918 -0.5834154 0.41049242 1.4623992 -0.6470828 -0.80110556 -0.842076 1.4063137 0.4064267 -1.4012467 -1.51156 -0.66073525 -1.2326764 1.3915304 -1.6097598 -1.1680654 1.3078756 -0.6685946 -1.5648686 1.0804002 1.5578338 -1.4532202 -1.4941028 -0.5354314 -0.39458364 -1.0363178 0.50652486 0.6863613 1.2986392 -1.4903429 -1.6850866 -1.7306389 -0.760653 -1.2985238 0.59533846 -0.5791006 -0.7715054 -1.2230364 0.7600007 -1.5599965 0.35744607 1.6492218 1.1998763 -0.65731704 -1.4319642 0.25371858 1.5374823 1.3772316 0.6156299; -0.6001503 1.3020393 0.6015257 -0.74280965 -0.8525441 -1.2765948 -0.8315411 1.455828 1.2261374 -0.6016622 -1.5240096 -1.4794228 1.4707756 0.765991 -1.3467996 1.5126883 -0.5253485 0.6893893 1.4125303 0.5867493 -1.3359665 1.5200851 1.4235941 1.3676026 1.2600791 -0.72988516 -0.71250767 1.3009957 1.6038842 -0.74951965 0.71653885 1.5103661 1.4429405 -1.4690909 -0.7010064 1.3032868 1.5358444 1.3413272 -0.7821238 -1.5454212 0.7787764 0.8569296 1.2669873 0.85702556 -0.72142833 0.53456086 0.8032131 -0.5928674 1.3807914 0.6352149 -0.9234446 -0.76086044 0.8775672 0.7683421 1.359658 1.5298544 0.608043 -1.57762 -1.5777211 -0.6482109 0.6155826 0.599254 0.54538584 1.4969337 0.6337292 -1.5118375 1.5805964 1.5059234 0.90468633 -1.6199198 0.589474 -1.4260241 -0.6756648 -0.8197054 1.4834093 0.68608606 -1.3820437 -0.62714505 -0.85912645 -1.5608077; 0.7351252 -1.3411615 -0.65652514 0.70317763 0.75120735 1.4182804 0.90264195 -1.4069332 -1.4954734 0.8526833 1.4787974 1.4274659 -1.4733374 -0.86540484 1.2155045 -1.4071176 0.63162386 -0.87842727 -1.4211329 -0.9066801 1.2517083 -1.4553312 -1.3347262 -1.244408 -1.4909606 0.80153215 0.58782446 -0.6248009 -1.5856631 0.78736085 -0.88172406 -0.8088403 -1.2640553 1.3023833 0.53766274 -0.8446387 -1.4818444 -0.8401068 0.7474025 1.4171759 -0.545915 -0.6299214 -1.3367907 -0.8236918 0.75366074 -0.85941094 -1.533051 0.8374226 -1.4758289 -0.8726239 0.75488883 0.8308918 -0.6180086 -0.89885235 -1.5784345 -1.4641479 -0.63735414 1.455295 0.7598565 0.6911981 -0.5948415 -0.59756595 -0.625059 -1.5393878 -0.80834043 1.3893615 -1.4575415 -1.3652066 -0.8211779 1.3131543 -0.6111043 1.2203242 0.8772422 0.7605943 -1.5329391 -0.6097236 1.5407095 0.6265247 0.7866342 1.5853405; -1.1992755 0.24142273 0.38509148 -0.06675783 -0.29082978 -1.9555964 -0.54884756 1.8931303 2.0054429 -0.1967174 -2.1283023 -0.6154092 2.068576 0.10089626 -1.8619429 2.16056 -0.5082204 0.1640636 0.7094224 0.33828893 -1.4248556 1.6985556 1.0585783 2.067172 0.5773746 -0.2870935 -0.6931922 1.6160338 1.365288 -0.6859497 1.2714567 1.7777119 2.158265 0.46748477 -0.29099777 2.0057063 2.380826 1.701722 -0.21562019 -2.045236 -0.41346732 0.24160114 1.5420064 0.9712443 -0.0575312 1.0693209 1.9422772 -0.7625267 2.1059291 0.85437995 -0.78231525 -0.47265658 0.50029474 0.09090349 2.2011755 1.4515787 1.3669615 -1.9434441 -1.6151391 -0.2595721 -0.9288923 0.5196026 0.50438994 2.212143 1.3649352 -2.1962368 2.0290797 2.0991447 1.4283711 -1.9449375 -0.02142606 -1.6500803 -0.9615869 -0.90909463 2.2303963 1.1204433 -1.2861799 -0.17796144 -0.8548195 -1.5939021; 0.86209565 -0.57867134 -1.4915959 1.2674674 1.3005455 0.74248374 1.2865865 -1.3965235 -0.76454204 1.4357047 0.78570646 0.5667839 -0.5377974 -1.3903422 0.676108 -0.89350826 1.3868601 -1.3020071 -0.8809912 -1.4398054 0.8837719 -1.2296047 -0.7033609 -0.6097102 -0.7177684 1.5454321 1.4654721 -0.78013915 -0.548687 1.3357031 -0.7780331 -0.77141184 -0.6940781 0.77476376 1.269183 -0.69450074 -0.81003386 -0.67357516 1.2124317 0.8029607 -1.3760451 -1.3367952 -0.8954946 -1.4265319 1.4547398 -1.5187204 -1.2396683 1.462066 -0.7122436 -0.7366073 1.5537275 1.2169203 -1.2688451 -1.3276469 -1.4542984 -0.53965896 -1.0982386 1.5980233 0.801495 1.3281662 -1.5206991 -1.2627743 -1.4024575 -0.66294974 -1.3739517 0.7327041 -0.5762563 -0.94455135 -0.5892286 0.6646615 -1.3499074 0.7558608 0.79488283 1.543499 -0.6998178 -1.3363292 0.6425479 1.3906351 1.3930656 0.9183623; -0.21472076 -1.2100593 0.87016875 -0.9005461 -1.1205655 1.5389899 -0.5610104 -0.90222466 -1.2504488 -0.96595305 1.4344174 1.3865498 -1.2785145 0.91891456 1.4362403 -1.3237911 -0.8528193 1.0309621 -1.098362 0.77807593 1.4679691 -1.3491994 -1.4724907 -1.4904892 -1.2813042 -1.2027755 -0.8911063 -0.6078787 -1.1209362 -0.5509318 0.070712894 -1.4307855 -1.3192751 0.94327146 -1.1907951 -1.3952231 -1.5594563 -1.8446426 -1.0396861 1.4577127 0.8801229 1.0680165 -1.4249582 0.61802536 -0.9610904 0.52079403 -0.05969026 -0.66287184 -1.4515682 0.5613374 -0.52480215 -1.076534 0.81699747 0.991957 -1.2719297 -1.1288062 -0.4653064 1.3755982 1.2902682 -0.8386 0.80791587 0.90870786 0.8869836 -1.3623577 -0.041749153 1.467511 -1.2179158 -1.2173984 -0.07478049 1.22546 0.83878183 1.1757222 -0.4470508 -0.71891814 -1.4664547 0.2953777 1.457532 -1.1116831 -0.7171359 1.3478038; 1.4787619 -0.7452127 -1.6565865 1.4669005 1.4506572 0.3390635 1.4602157 -0.591054 -0.5378577 1.7931017 0.5912888 0.33937803 -0.7490883 -1.5101676 0.510628 -0.84564006 1.4185089 -1.6928172 -0.41889104 -1.5085478 0.355088 -0.4148381 -0.46870986 -0.61659884 -0.5094862 1.8672895 1.399647 -1.198019 -0.1336765 1.4542279 -1.4988999 -0.6694809 -0.75400734 0.17313424 1.4677774 -0.69405025 -0.46691328 -0.63721377 1.4838572 0.6535087 -1.2991897 -1.6326739 -0.32476538 -1.328525 1.5947293 -1.5617783 -1.2525754 1.2731571 -0.5622408 -1.4715666 1.3098936 1.6200558 -1.4401386 -1.6638999 -0.6945731 -0.3916924 -1.2267212 0.76315063 0.62713426 1.5984977 -1.5874518 -1.4652865 -1.6580235 -0.6630639 -1.4841142 0.5394364 -0.62859786 -0.49355066 -1.2026439 0.30983984 -1.7901484 0.56393975 1.494579 1.2189261 -0.5524439 -1.2193135 0.051625438 1.6843454 1.4920427 0.27460408; 0.546035 1.00429 -1.0811478 1.0468713 1.0058533 -1.3394121 0.6134614 1.4455358 1.3578025 0.92319596 -1.4780815 -1.3762451 1.3917716 -0.9205032 -1.3715576 1.6316353 0.7977219 -0.8791075 1.1835726 -0.80351305 -1.0925002 1.3930087 1.3606663 1.7223952 1.1379505 1.0201464 0.95006615 0.002785635 1.094376 0.82627934 -0.5304114 1.5490252 1.6215005 -1.0732431 1.0311942 1.669274 1.5852101 1.3932353 1.0172732 -1.3759012 -0.8395477 -1.165258 1.349238 -0.6172104 0.9707679 -0.60709447 0.30802116 0.65929055 1.618256 -0.8392621 0.15212217 0.8041744 -0.77076536 -0.7813335 1.644244 1.2310232 0.43757612 -1.5556964 -1.4252837 0.99467474 -1.2341671 -1.1035417 -1.1194257 1.3029109 0.027459014 -1.5342244 1.2049361 1.4554105 -0.012382822 -1.2591363 -1.097741 -1.1367574 0.6536394 0.75300384 1.5369787 -0.46335074 -1.2133739 0.9317129 0.86346066 -1.2900668; -1.4700692 0.7079237 1.4088513 -1.3329743 -1.5303141 -0.5660675 -1.4215585 0.685863 0.6279434 -1.4271814 -0.66046214 -0.7510874 0.64718163 1.2539415 -0.8612717 0.8094879 -1.3679512 1.4132532 0.8352006 1.3060861 -0.8360928 0.724448 0.756878 0.72646314 0.6277992 -1.402467 -1.2378585 0.8500109 0.8240343 -1.2982714 1.1960906 0.55038404 0.5355573 -0.8449297 -1.2612144 0.60600513 0.83539015 0.8146846 -1.2577635 -0.61986864 1.36585 1.6017075 0.7240413 1.4311937 -1.5427843 1.2764466 1.5205014 -1.3277398 0.76998425 1.3451717 -1.2978464 -1.3135037 1.4935933 1.3188162 0.6719917 0.55275375 1.1471598 -0.7346106 -0.66290903 -1.5166167 1.2983046 1.421518 1.3407732 0.8014245 1.3674018 -0.626535 0.86233926 0.90008205 1.3869396 -0.8010883 1.4569772 -0.53608805 -1.458565 -1.4916842 0.59574604 1.4553672 -0.65735906 -1.6145412 -1.4411672 -0.6925777; 0.1402105 -1.1751865 1.1200871 -0.9041539 -0.8241411 1.592135 -0.39971188 -0.9502282 -1.0344731 -0.94492847 1.5313339 1.4160669 -0.98415077 0.72784835 1.2586603 -1.1976366 -0.7476526 0.90585274 -1.346399 0.6562959 1.3248485 -1.2607503 -1.537081 -1.2570192 -1.311253 -1.1455367 -0.08398105 -0.5813777 -1.5827639 -0.45330355 -0.02603817 -1.2366409 -1.448494 1.0162725 -0.6797445 -1.0118662 -1.3009592 -1.0523132 -0.7180462 1.3781708 1.2015904 0.8850056 -1.4038161 0.1698773 -0.94782 0.25199416 -0.3741613 -0.08427965 -1.1790301 -0.20515995 -0.1474616 -0.39816734 0.4310156 0.67480814 -0.93891126 -1.5888172 -0.34927207 1.1857289 1.2785667 -0.8826221 0.8964506 0.7157902 0.4017357 -1.319969 -0.124715626 1.3798187 -1.4468914 -1.1717533 -0.19161046 1.6088115 0.9287854 1.295954 0.07455002 -0.24917108 -1.3579708 -0.17827585 1.668072 -1.1768835 -0.32554504 1.3965337; 0.95938784 0.30842486 -1.1332842 1.3405093 1.095394 -1.033965 1.4334751 -0.23218271 0.8505446 1.0774091 -1.0753176 -0.6986084 0.5054943 -1.3654372 -0.99653184 0.9217959 1.4010837 -1.1258885 0.7077516 -1.3172432 -1.0523331 0.58651537 0.955203 0.7665223 0.70952445 1.1085991 1.0708859 -0.32196778 1.0369745 1.1396123 -1.0037153 0.40620336 0.9347744 -0.74922806 1.2163742 0.44235682 0.90429205 0.4176606 1.2506093 -0.89140457 -1.3472393 -1.3848189 1.144114 -1.3345022 1.197141 -1.1451658 -1.147777 1.3152927 0.67529154 -1.0916102 1.3945345 1.3632321 -1.2694364 -1.5074518 0.64526254 1.1482381 -0.96483886 -0.6288483 -0.3181727 1.258871 -1.5364226 -1.144836 -1.1653447 1.0401307 -1.104703 -1.177053 1.0029695 0.77905554 -1.0336434 -1.0848076 -1.290993 -0.7995239 1.291319 1.3115921 0.8467178 -1.2549441 -1.1100935 1.253715 1.3815309 -0.8114878; 1.6028994 -0.7627015 -1.837188 1.6174327 1.569211 0.33139494 1.3589587 -0.68904275 -0.65984446 1.5365179 0.3871925 0.5108961 -0.6941516 -1.4778458 0.32773224 -0.71212685 1.7226362 -1.5823416 -0.45726013 -1.5494692 0.22907965 -0.6903859 -0.39706752 -0.7311749 -0.61683625 1.5232389 1.6688504 -1.4519119 -0.19140093 1.5946246 -1.2712039 -0.61055 -0.63979447 0.026876535 1.6759057 -0.58950454 -0.6570951 -0.63442695 1.4261944 0.37100276 -1.4338604 -1.7475884 -0.4344113 -1.371622 1.587442 -1.3544942 -1.2854116 1.596695 -0.6012546 -1.4852115 1.218135 1.4092271 -1.4070934 -1.6641662 -0.4670602 -0.23819362 -1.4121758 0.5842433 0.7331623 1.4117314 -1.5508126 -1.6748337 -1.7895681 -0.6528225 -1.1893504 0.4038492 -0.59638846 -0.52901536 -1.1914638 0.59543806 -1.4803618 0.36825928 1.4366792 1.2644506 -0.48879552 -1.5168002 -0.012997666 1.7392904 1.5350274 0.4170016; -1.3142703 0.9208064 1.4463264 -1.437983 -1.3069 -0.7360153 -1.5190207 1.4446421 0.9182159 -1.5723677 -0.8280387 -0.79984516 0.5703915 1.4478467 -0.79330236 0.6593723 -1.5106033 1.3531843 0.77019006 1.4530599 -0.651374 0.7868688 0.8585514 0.8522937 0.7266699 -1.2707263 -1.395306 0.89221305 0.7695917 -1.5683345 1.5291911 0.67027193 0.89250934 -0.6311343 -1.5118747 0.6662197 0.64016485 0.76705366 -1.4793178 -0.8245103 1.3607539 1.5964557 0.8755778 1.472088 -1.2647517 1.5377076 1.5414076 -1.5612676 0.5287296 1.3436164 -1.3677801 -1.5242382 1.2629088 1.2871665 0.8634263 0.55668366 1.5718397 -0.6601184 -0.71681964 -1.4698944 1.5564252 1.3747991 1.312879 0.8010356 1.4459617 -0.6311336 0.78741586 0.5369593 1.3711842 -0.53252965 1.4594709 -0.7266037 -1.2214365 -1.2870944 0.6475699 1.3072386 -0.607208 -1.4521683 -1.2855394 -0.76625717; 0.4640434 -1.417384 -0.64761066 0.76374245 0.4586066 1.6000997 0.50228316 -1.0534751 -1.3517238 0.6499357 1.4653037 1.5519565 -1.4729865 -0.5934324 1.5173811 -1.4795266 0.52873784 -0.39500767 -1.5460026 -0.72027236 1.5643826 -1.2119939 -1.5910827 -1.2546834 -1.2623354 0.38458452 0.7394973 -0.9925015 -1.5802742 0.5947039 -0.6026399 -1.1306304 -1.2477719 -0.06713344 0.5935117 -1.075184 -1.33885 -1.366883 0.42485443 1.4403555 -0.3372899 -0.47826788 -1.4726009 -0.73519087 0.62897575 -0.8287793 -0.51349837 0.68735605 -1.2379507 -0.7736261 0.8370858 0.5781401 -0.7966051 -0.57961076 -1.1709231 -1.5191797 -0.6322326 1.4269975 1.1552559 0.44062284 -0.59227455 -0.55403274 -0.52384996 -1.5210522 -0.6232214 1.4053361 -1.4381019 -1.5132126 -0.5558346 1.5253316 -0.35716113 1.3042078 0.70336086 0.6972781 -1.54331 -0.6195023 1.5960926 0.54958874 0.6511526 1.2677027; 1.3252639 -0.81805605 -1.6052835 1.4787786 1.3797725 0.7381072 1.5072078 -0.7965542 -0.7173441 1.3359805 0.6207421 0.82299733 -0.7716996 -1.4303868 0.8480263 -0.69885933 1.3155062 -1.3939395 -0.5974395 -1.4282236 0.65979505 -0.5578796 -0.5426088 -0.5673822 -0.6326606 1.681714 1.3987265 -0.7584227 -0.6867844 1.5790428 -1.2712256 -0.72627467 -0.692618 0.63916475 1.4887189 -0.69749516 -0.6422329 -0.66318667 1.4237177 0.48882112 -1.442053 -1.6785512 -0.6104426 -1.5716821 1.3905746 -1.3528746 -1.4809866 1.2378525 -0.6029412 -1.3607173 1.3234413 1.3298439 -1.6120102 -1.5191318 -0.63540274 -0.62373435 -1.2910507 0.818557 0.6447414 1.3376123 -1.4815527 -1.3967742 -1.3158073 -0.691419 -1.4153429 0.8621509 -0.79558754 -0.61078495 -1.3128165 0.75107884 -1.4190214 0.86461914 1.3209497 1.4035666 -0.6037165 -1.4143502 0.47100872 1.3369278 1.5226852 0.67135173; -1.398747 0.2919547 1.3287765 -1.2977244 -1.5413752 0.49146912 -1.0906819 0.20707074 0.05576584 -1.5708607 0.42506477 0.62023306 -0.092261784 1.2735246 0.5289542 0.154849 -1.6139785 1.2281322 -0.28401512 1.4352485 0.7806948 -0.21933629 -0.28657898 0.14616889 -0.23745759 -1.2475854 -1.4709464 0.7852765 -0.88663566 -1.4025534 1.3588942 0.3581037 -0.21325059 0.6966235 -1.5354728 0.55450016 -0.17281906 0.53557473 -1.244069 0.60572636 1.0883187 1.4295005 -0.6449253 1.4436097 -1.4687897 1.5116885 0.31781766 -1.1187443 0.49786657 1.3003434 -0.76684743 -1.481562 1.6138107 1.6046491 -0.17685896 -0.89002854 1.0384339 0.24995607 -0.30832335 -1.453335 1.2878966 1.4541708 1.3201561 -0.008218709 0.78672785 0.16906421 -0.4030775 0.0836047 1.0200409 0.42204309 1.537037 0.64535654 -1.4887222 -1.3327788 -0.085875556 0.9911219 0.7072304 -1.4524639 -1.4154874 0.4672165; -1.1801074 0.10827604 1.3593593 -1.4428692 -1.3301395 0.4208311 -1.5836873 0.36246735 0.20938605 -1.273383 0.542009 0.5115681 0.065566085 1.5344706 0.3882704 0.024239238 -1.3758698 1.3168546 -0.13341323 1.5704827 0.58462524 0.34407866 -0.553948 -0.053311817 0.08585957 -1.3519237 -1.3794181 0.2294325 -0.63378817 -1.5437403 1.1458663 0.27944133 -0.16333011 0.40375715 -1.3423983 0.051796272 -0.02549914 0.25055268 -1.5206953 0.5207409 1.387792 1.6507744 -0.6543356 1.7076644 -1.3430072 1.4921654 1.2702075 -1.4380188 -0.111999 1.2354949 -1.4471866 -1.6532183 1.3863308 1.4731745 0.19008815 -0.7147677 1.2337775 -0.21733405 -0.1942667 -1.3554841 1.69283 1.3335401 1.6883523 -0.2802176 1.0879492 0.33436003 -0.28127462 -0.026228426 1.0701648 0.4314554 1.3932071 0.49679714 -1.451592 -1.4175411 -0.42390114 1.4703277 0.8296218 -1.4751151 -1.3796 0.43332487; -1.4839673 0.7298338 1.4306898 -1.6133889 -1.3512832 -0.74796796 -1.567353 1.2957053 0.88858235 -1.481291 -0.6483519 -0.79763746 0.7535622 1.432207 -0.8361447 0.70834106 -1.3444861 1.3636864 0.70888054 1.4353689 -0.83449525 0.8448384 0.715601 0.8226689 0.7515313 -1.5390816 -1.3130659 0.5654493 0.86256397 -1.4045364 1.2658764 0.8429968 0.54763234 -0.22935164 -1.6420934 0.76785576 0.7925096 0.6687123 -1.4399163 -0.74509054 1.6156147 1.5155337 0.5808229 1.6031859 -1.3955942 1.3849488 1.4373943 -1.5429393 0.6180809 1.2309693 -1.4606413 -1.5530301 1.36047 1.488613 0.79083145 0.86350983 1.3408023 -0.95518374 -0.73950833 -1.6069915 1.5267267 1.4317788 1.3237573 0.79766697 1.4061856 -0.68142503 0.54287636 0.8499858 1.3368359 -0.5203003 1.6067415 -0.8138018 -1.2806835 -1.4595369 0.85819525 1.4457823 -0.6927253 -1.4128561 -1.5235391 -0.82534266; 0.5871434 -1.516347 -0.6672987 0.59266037 0.67488533 1.5154822 0.7707769 -1.5182358 -1.566264 0.8626015 1.4624463 1.3374503 -1.5052382 -0.5655866 1.259055 -1.4256045 0.69853306 -0.77625674 -1.2901492 -0.6384367 1.4761971 -1.2324238 -1.3586473 -1.3556198 -1.2765132 0.7180247 0.5916098 -0.50456417 -1.534407 0.65521824 -0.76649046 -1.3562199 -1.5476301 1.7190359 0.85064244 -1.1577711 -1.2342955 -0.70854896 0.84158623 1.5317819 -0.6604878 -0.79778576 -1.4618531 -0.85904074 0.6472979 -0.55178 -1.1095732 0.8509222 -1.4201745 -0.77626926 0.62543905 0.8932071 -0.7597324 -0.7039053 -1.2457987 -1.3850842 -0.77351636 1.2371114 1.18628 0.582334 -0.8070585 -0.83901614 -0.6990574 -1.2780463 -0.8675855 1.2115436 -1.2865357 -1.336518 -0.5351758 1.5390893 -0.6801036 1.6000217 0.6532078 0.77693266 -1.5806444 -0.6541071 1.5791429 0.7847743 0.58969223 1.3991976; 0.8254299 -0.03210773 -0.520543 0.056555144 -0.0010850957 -0.26868546 -3.9864597 2.845261 0.015323215 0.3127961 -0.30904365 -1.17569 0.85204744 1.1149427 -0.22293155 -1.7921475 0.16858166 -0.64136267 1.1046933 1.3118753 -2.01371 1.645314 -0.14166512 -2.6071157 0.07749194 0.44544452 1.2471545 -1.0566155 2.3802269 1.2771112 -0.357448 -1.7488787 1.6797291 -0.65848386 0.097014695 -1.7348871 -1.52453 -2.2147105 -1.3247643 -2.0108159 1.2467495 0.2577444 2.0316803 1.4767653 -0.44863272 -0.12952596 4.397411 -2.736767 -3.213499 -1.0338287 -5.02574 -1.364647 -0.23029311 1.3929274 1.4470334 1.9194506 2.9644063 -2.5542555 2.4035604 -0.89115566 1.7654423 -0.4096585 -1.1186347 -0.8560379 3.1485672 0.98083234 0.84771925 -0.018585451 1.1513308 -1.3859401 0.37354326 -2.3149326 1.5453881 -3.916901 -1.3877796 1.8227706 -2.0606222 0.19815202 -2.4940932 -1.4098415; -1.2405125 0.8883526 1.2941188 -1.4080857 -1.288617 -0.6732841 -1.4168239 1.3923492 0.63699347 -1.2701285 -0.8625427 -0.70757306 0.8243495 1.4300717 -0.90404415 0.8512369 -1.4590875 1.3592882 0.77814686 1.4220839 -0.7259504 1.0087124 0.58935887 0.6514544 0.682051 -1.3832349 -1.5803943 0.5043146 0.74286735 -1.5422146 1.5463047 0.6448947 0.9284407 1.0649769 -1.621415 0.73927176 0.5079813 0.5555981 -1.4407628 -0.6249019 1.7390127 1.3615699 0.8500029 1.3741145 -1.4209248 1.3587803 1.4090064 -1.4859393 0.8134012 1.5471542 -1.4644153 -1.6152682 1.2934558 1.5939217 0.93333447 0.6631744 1.5487037 -0.8660424 -0.5353327 -1.450653 1.6374456 1.5407143 1.2603612 0.7544321 1.5600632 -0.65179884 0.90027636 0.60164243 1.4661611 -0.88663954 1.7017314 -0.77998734 -1.5039632 -1.2734084 0.583993 1.3495048 -0.6855622 -1.6524172 -1.3674508 -0.6981209; -1.3709785 0.7290726 1.566041 -1.417345 -1.3838332 -0.8518878 -1.2611355 0.8886508 0.60368794 -1.2953051 -0.76320404 -0.68119353 0.65209717 1.2893888 -0.5856188 0.7653463 -1.4049611 1.538009 0.9118716 1.253846 -0.63833994 0.77901095 0.76680297 0.57388854 0.6095529 -1.5832373 -1.5614848 1.4536958 0.5463614 -1.5283797 1.5054077 0.63618344 0.8918866 -0.6163817 -1.2728872 0.86909413 0.6673285 0.70722955 -1.2690592 -0.5511513 1.2688532 1.260911 0.6123143 1.3417791 -1.4176133 1.5734608 1.4041395 -1.2254267 0.91483766 1.495124 -1.380125 -1.2747412 1.4792655 1.3878962 0.7463691 0.6297858 1.3207867 -0.81040883 -0.7270633 -1.2717121 1.4287293 1.436046 1.2576407 0.56478244 1.3548148 -0.7442435 0.6555745 0.9301573 1.3640987 -0.6567407 1.58974 -0.55449986 -1.6135024 -1.4586785 0.7606377 1.3692601 -0.6524596 -1.5029373 -1.3457539 -0.7698351; -1.3260009 0.6216436 1.4413024 -1.2771066 -1.3621124 -0.589062 -1.0885608 0.67908114 0.6723583 -1.232831 -0.8705094 -0.7800691 0.61713743 1.3201637 -0.6199351 0.8947537 -1.4196883 1.4479334 0.8400723 1.29009 -0.531255 0.76478285 0.8206809 1.0645777 0.8519468 -1.3946192 -1.4264334 1.4114486 0.76139456 -1.2639837 1.1960552 1.0427336 0.49806812 -0.9015305 -1.4063623 1.323381 0.8492325 1.226955 -1.4798311 -0.7079484 1.3670071 1.2958125 0.71859914 1.4346013 -1.18971 1.2347355 0.6881592 -1.3712177 1.3716234 1.2366865 -0.51271373 -1.309223 1.4522474 1.3505993 0.5685342 0.70345265 1.141695 -0.7332535 -1.243152 -1.4385515 1.3530679 1.3191876 1.4118713 0.83035374 0.98828924 -0.73661196 0.59262985 0.5062029 1.073693 -0.7665148 1.5083994 -0.6320169 -1.4875761 -1.0524687 0.79975134 1.1579223 -0.6352189 -1.5550815 -1.1854603 -0.8309619; 1.3372233 -0.5933916 -1.3403486 1.2569575 1.525487 0.75406426 1.2725517 -0.6264959 -0.68624336 1.6181093 0.58234817 0.8033897 -0.6807448 -1.5823251 0.87256336 -0.54909146 1.3237832 -1.2901617 -0.7342386 -1.2480867 0.57802016 -0.8686513 -0.81933093 -0.89891833 -0.8855545 1.5405948 1.5846949 -1.3531945 -0.664567 1.2452879 -1.4874206 -0.55154294 -0.6437451 0.8719909 1.2794636 -0.84180456 -0.67135763 -0.9036877 1.3856657 0.74260485 -1.2692255 -1.4605838 -0.708457 -1.4660748 1.4534013 -1.2971675 -1.2769006 1.2247305 -0.88639444 -1.4132003 1.4561772 1.229423 -1.3234162 -1.239993 -0.7263189 -0.62607664 -1.4652314 0.6068709 0.8548323 1.2360119 -1.2424881 -1.238944 -1.29358 -0.68728095 -1.3418815 0.73402303 -0.9101596 -0.804112 -1.4430062 0.5628348 -1.5822634 0.60471827 1.4912568 1.3665897 -0.8291608 -1.396662 0.6003831 1.3413944 1.5556196 0.8068221; 1.1373682 -0.041856386 -1.2375833 1.3606168 1.394198 -1.1156107 1.6288315 -1.1307555 0.4045951 1.4737607 -1.0987226 -0.8301538 -0.0168942 -1.5047834 -0.9040739 0.6228362 1.1812602 -1.3410797 0.71510404 -1.5495502 -0.9411237 0.22297242 0.98969835 0.7661019 0.3892552 1.4689041 1.469157 -0.020616584 0.97876334 1.4631218 -0.95388144 0.29736492 0.48543897 -1.2117648 1.2933186 0.33797514 0.9104994 0.47448257 1.3075773 -0.8862437 -1.43968 -1.52823 1.1106466 -1.4956005 1.4584321 -1.5215492 -1.1393311 1.3221518 0.77982885 -0.8299391 1.553834 1.4814109 -1.2856648 -1.4068128 0.10775949 1.2733426 -0.7986203 -0.29343817 -0.55885243 1.5541469 -1.4096519 -1.174366 -1.336999 0.8845176 -1.0983423 -0.94782865 0.783133 0.34320062 -1.0506841 -0.7218255 -1.1801816 -0.97307986 0.9942803 1.5297232 0.7726467 -1.1085551 -0.94080234 1.4254388 1.3421493 -0.8311276; 1.0456917 0.10000325 -1.0647655 1.3630224 1.5062686 -0.8519928 2.2177775 0.47145864 0.6041935 1.3900571 -1.1128203 -0.6332339 0.413864 -1.644106 -1.062236 0.69155496 1.3573989 -1.1914474 0.6716149 -1.6037724 -1.0383962 0.4931178 0.6411502 0.19841103 0.5902315 1.0859153 1.2941445 -0.8820091 0.78183997 1.2467078 -1.2359407 -1.0199217 0.75499964 -0.92330074 1.3109261 -0.60939384 1.3381318 -0.92044806 1.3549138 -1.192548 -2.0128658 -1.5460229 0.7707398 -1.5070846 1.4875774 -1.3678889 -0.36291566 1.736191 0.06489671 -1.3358643 1.8496311 1.5064573 -1.5769559 -1.4568774 0.84443516 1.0372952 -1.3171339 -0.8924025 0.6949091 1.6652921 -1.8470101 -1.3205456 -1.2306036 0.7260081 -1.4671773 -1.0823413 0.8508939 0.87117255 -1.0851597 -1.0937892 -1.3917534 -0.8163214 1.1721792 2.038497 1.1265012 -1.3513696 -1.0664613 1.4649643 1.5306622 -1.0002198; -1.3146498 0.8177205 1.6742293 -1.701654 -1.6864641 -0.6176133 -1.3850754 1.5909084 0.8140019 -1.6016473 -0.6693593 -0.73169196 0.7797324 1.5380907 -0.7682851 0.57718456 -1.4221773 1.7000674 0.80550617 1.3977965 -0.66839975 0.8412349 0.72568196 0.7378436 0.82716984 -1.6406165 -1.404006 0.8551212 0.62001985 -1.4756765 1.5034657 0.84879756 0.72187257 1.1373383 -1.4632319 0.7850371 0.72103345 0.54805684 -1.3791653 -0.7471072 1.5743697 1.5506662 0.8492095 1.271077 -1.3604497 1.4604138 1.5579184 -1.4266587 0.5996709 1.4410279 -1.3464365 -1.4533806 1.3252648 1.3290216 0.72477305 0.69853663 1.594985 -0.84718305 -0.7689755 -1.3331273 1.4239664 1.3383309 1.326341 0.6062739 1.6217992 -0.7128614 0.760859 0.8524483 1.4438844 -0.7103318 1.6367087 -0.80248415 -1.3546913 -1.3452601 0.7497708 1.4698697 -0.62974226 -1.4116836 -1.6186969 -0.5222256; -1.4241123 0.40394202 1.6015515 -1.5425543 -1.5478983 -0.65832967 -1.3953434 0.6278642 0.6541539 -1.6724092 -0.5239876 -0.57907355 0.74413544 1.4001136 -0.46262112 0.82478124 -1.6496305 1.7276844 0.36678764 1.3446504 -0.4386818 0.65536577 0.30924243 0.6275998 0.6870785 -1.458617 -1.3503997 0.81569046 0.36469588 -1.4411186 1.4117558 0.5422668 0.7353189 -0.24267916 -1.4583411 0.56398433 0.7431615 0.67532885 -1.6368809 -0.43738478 1.5290258 1.5364151 0.3027816 1.5810137 -1.4836869 1.3127866 1.2161776 -1.2443923 0.5430292 1.4415227 -1.2237272 -1.433827 1.5586858 1.3310709 0.6827317 0.2359913 1.3291578 -0.54571044 -0.47282782 -1.5220164 1.5208663 1.6409497 1.6513189 0.44634423 1.3250858 -0.7068654 0.70571417 0.5090213 1.0443252 -0.5036369 1.5739874 -0.47700346 -1.4080169 -1.4544952 0.69719994 1.3903166 -0.1620882 -1.5214784 -1.5648146 -0.5961047; -0.8784016 1.4195023 0.685415 -0.8053105 -0.9160003 -0.85498697 -1.499421 1.5901269 0.702776 -0.5694718 -1.2865741 -1.5649686 1.2774689 1.4036273 -1.2634877 0.73801845 -0.6717844 0.6576119 1.5077052 1.5216151 -1.3136766 1.2519131 0.8013376 0.79130733 1.5381513 -0.73935694 -0.531395 0.9022971 1.3844502 -0.6983086 0.56747764 0.6564996 1.349907 -1.5241218 -0.6629791 0.8362133 0.5612187 0.4983117 -1.243773 -1.5937401 0.708492 0.86556506 1.2927604 0.8686698 -0.7322058 0.80958533 1.5274204 -1.5079627 0.5939777 0.8746869 -1.3193904 -1.3475173 0.60713524 1.4535197 1.3223035 1.3188736 0.7434825 -1.4040155 -0.7218386 -0.78282344 1.5834727 0.7387902 0.56622493 0.62891287 1.3117449 -0.79820585 1.2423048 1.0174693 0.7450019 -1.2764401 0.5144896 -1.343054 -0.881383 -1.4596007 0.694354 0.61531186 -1.440599 -0.5261466 -1.2762765 -1.4660087; 0.57487875 1.2025236 -1.0077935 0.8937829 0.9498702 -1.3490664 -0.45673624 1.657042 0.64043367 1.1967596 -1.3958273 -1.6003493 1.0998737 -0.592199 -1.069518 0.36534914 0.7347366 -1.147255 1.6761553 -0.5883228 -1.5673125 1.4499098 1.2449985 -0.39605832 1.3854842 1.2450368 1.0499696 -1.0891062 1.5997663 0.8149255 -0.54304636 -0.14337966 1.4526213 -1.626596 0.81542 -0.42623174 0.8576327 -0.32789502 0.526585 -1.6902487 -0.88270444 -0.91512746 1.4342588 -0.18535988 0.6690508 -0.4181286 1.4846117 -0.2075379 -0.18973997 -0.73516476 -1.59776 0.5307642 -0.75794435 -0.29317936 1.4678812 1.5517911 0.46779898 -1.823979 0.46344948 0.80921394 -0.35909602 -1.0345505 -0.9316745 0.8114609 0.39137512 -0.6237467 1.1991745 0.87256354 -0.0071740155 -1.6029409 -0.8339492 -1.5860085 0.8175971 -0.47997117 0.72879744 0.12551317 -1.6350013 0.962611 -0.06826853 -1.6563518; 1.4602132 -0.54078966 -1.633402 1.7936801 1.5067625 0.38824797 1.4482813 -1.2597471 -0.52337325 1.7545971 0.5611254 0.5053578 -0.550477 -1.5164545 0.3005062 -0.85685474 1.5038965 -1.6711961 -0.26852292 -1.5429138 0.3823142 -0.72745967 -0.4578424 -0.44967487 -0.36225417 1.8222697 1.4426239 -0.8833786 0.03375023 1.3812159 -1.2110516 -0.30709466 -0.6570503 0.20449862 1.761935 -0.4237048 -0.8397489 -0.44401193 1.6244138 0.7871213 -1.0381955 -1.6155616 -0.22176096 -1.2675654 1.4735922 -1.5465499 -1.323115 1.4312681 -0.38585892 -0.7099149 1.5779499 1.3955384 -1.411323 -1.3672303 -0.88073933 -0.33794746 -1.3074173 0.48353592 0.56487846 1.7294471 -1.4518702 -1.555366 -1.4002308 -0.7121247 -1.2753875 0.67356217 -0.6238532 -0.8022213 -1.1540176 0.5510105 -1.1765938 0.6121335 1.0185057 1.5954785 -0.50013155 -1.1842767 0.084610805 1.6074989 1.3629642 0.22420858; 0.8960022 1.6099064 0.41934013 -1.048452 -0.9901957 -0.38374317 -2.697588 1.1579268 -0.2220063 -0.77759707 -0.14675887 -1.9186121 0.85027283 2.3985412 -0.4853837 -0.8125131 -0.7543351 0.6755132 2.3112843 1.9671038 -1.2472501 1.7898663 0.48865795 -0.09147664 1.5451126 -0.86207354 1.0488272 -1.0981926 1.4278088 1.0830939 -0.4347316 0.7104395 1.3118553 0.28339022 -1.5627596 0.5942534 -0.7925151 0.5700653 -2.3456862 -1.4904339 2.794345 1.4879236 2.1062686 1.7066091 -1.9409131 0.11618564 1.2431635 -2.4873822 0.07667487 -1.3230243 -1.9350528 -2.3131504 0.90526044 2.3400147 0.93248326 1.3138856 1.1519037 -2.0155773 -0.74828243 -1.840284 2.6355882 0.49067175 -0.5997396 0.09558321 1.392012 0.9870486 0.19456503 -0.26505616 0.060479604 -1.8989125 2.7045217 -1.9540992 0.98491895 -2.4097538 -0.72662526 1.2393289 -1.0014642 -1.1884094 -1.6481175 -2.0200226; 0.7720131 -1.3193558 -0.6981519 0.4541869 0.6506332 1.5635084 0.57767296 -1.2234809 -1.341169 0.48896065 1.4104661 1.2769125 -1.1326444 -0.5533199 1.4282125 -1.1233287 0.46810728 -0.48445722 -1.3117211 -0.47517303 1.5651762 -1.3519675 -1.4870108 -1.4575094 -1.251273 0.36933243 0.69917977 -1.0910363 -1.5710241 0.6425877 -0.78052324 -1.3753759 -1.489305 -0.44293818 0.57410574 -1.1327777 -1.3420784 -1.3303144 0.5376006 1.4074496 -0.64299697 -0.6288386 -1.5919871 -0.5771542 0.7667376 -0.4652833 -0.5212769 0.8386251 -1.1818584 -0.4870202 0.75717306 0.7054547 -0.74090767 -0.73129106 -1.1691436 -1.3792727 -0.8172737 1.351413 1.3971809 0.6818658 -0.69239455 -0.53774697 -0.7910479 -1.391425 -0.7856567 1.2196476 -1.4313874 -1.3625444 -0.75967264 1.5127728 -0.41766906 1.2632955 0.73698145 0.56083155 -1.2695321 -0.5384243 1.5777631 0.39521298 0.7784688 1.5602076; 0.70071363 -1.172952 -1.4732575 1.3863753 1.430489 -0.773316 1.8872778 -1.7109294 0.06955378 1.49581 -0.86151236 0.2109145 -0.54097176 -1.6159319 -0.49706542 0.5363239 1.5449283 -1.491588 -0.8306031 -1.8790454 -0.70174176 -0.95943695 0.742217 0.27226076 -0.7048444 1.5508997 0.9702262 0.9623347 0.56488484 1.0257416 -0.3110908 0.26361296 -0.42232573 -0.9869222 1.6405641 0.003232886 0.7447732 0.2088163 1.6187224 -0.32023054 -2.030804 -1.479632 0.12346192 -1.7153685 1.762613 -1.4349687 -1.685978 1.7496316 0.22679125 -0.25032112 2.1639025 1.8995203 -1.4527646 -1.7803264 -1.4283701 0.6230321 -1.0771611 1.1387138 0.0735191 1.6669295 -1.8609587 -1.4866186 -1.1122175 0.22707282 -1.524134 -0.6538129 0.39095187 0.18662786 -0.8888589 0.1855313 -1.7954779 -0.13958672 0.21107891 1.7241561 0.68386865 -1.438519 -0.83084816 1.6179576 1.5958068 -0.3468905; -0.78268945 -0.18585452 0.9366798 -0.3741686 -0.29985937 0.10092732 2.8518305 -2.7070687 0.13811992 -0.42872018 0.33925232 1.1157906 -0.74885917 -1.0645204 -0.094508775 1.3768414 -0.294069 0.7221955 -1.1158714 -1.3162255 1.7473469 -1.6203088 -0.22942218 2.897442 -0.29865187 -0.57103163 -1.4039772 0.91319585 -2.105201 -1.1814395 0.6823928 1.3027991 -1.4194951 0.28157973 -0.0282666 1.1367319 1.0263174 1.4730511 1.048552 1.8240714 -0.60223234 -0.16033284 -1.9955322 -1.1472833 0.18955219 0.5065148 -3.175753 2.4870794 3.3076198 1.3172089 4.719128 1.4232363 0.16961266 -1.6117142 -1.2506185 -1.9737716 -2.572746 2.4879816 -1.6411375 0.87891144 -1.1917274 0.61382496 1.1270877 1.1721736 -2.3796558 -0.90037125 -0.60098153 0.3115898 -0.38138425 1.223893 -0.39217997 2.2345538 -1.434488 3.5539136 0.8487053 -1.3142956 2.1698837 -0.49509165 1.9066361 1.5298519; -1.6154653 0.70185363 1.4932618 -1.7526498 -1.6550848 -0.7000157 -1.3835855 0.78419 0.6373169 -1.5075163 -0.55234987 -0.42786577 0.80664694 1.2709407 -0.7376188 0.9527686 -1.4624469 1.5575477 0.54770464 1.328575 -0.47974008 0.597521 0.44841605 0.87055874 0.48390523 -1.5837631 -1.4570035 1.1496274 0.03452988 -1.6121384 1.1576126 0.7472577 0.42083 -0.35826233 -1.5085453 0.8598873 0.8404838 0.87914973 -1.6486082 -0.5127787 1.0003597 1.4087158 0.48522645 1.3531518 -1.508441 1.5922153 1.2722998 -1.2903574 0.71905035 1.2486812 -1.277921 -1.4777898 1.426226 1.3316289 0.78159094 0.4111799 0.9876908 -0.41248038 -0.58471256 -1.3972613 1.2203617 1.5534158 1.7420676 0.54418856 1.2550851 -0.7058661 0.4282304 0.59375334 1.1276628 -0.54924375 1.5128691 -0.49083915 -1.3662071 -1.1048181 0.82206154 1.2459466 -0.40869883 -1.6809613 -1.2389501 -0.48629168; 1.4755853 -0.7049386 -1.2191061 1.498819 1.3772156 0.7661652 1.2466971 -1.5238296 -0.9039683 1.3591284 0.76231015 0.6823476 -0.8622641 -1.5477117 0.60857683 -0.64954937 1.2847375 -1.5766475 -0.75617963 -1.2846887 0.7297807 -0.80634415 -0.76909953 -0.8155229 -0.71925133 1.3998184 1.5190144 -0.5177676 -0.8569759 1.2411135 -1.5619336 -0.887031 -0.823621 0.6793635 1.2868497 -0.7375916 -0.6645191 -0.5387365 1.5949868 0.674987 -1.5119795 -1.4732814 -0.7778439 -1.4182494 1.2282778 -1.3126019 -1.3678665 1.2827349 -0.5519139 -1.303548 1.370609 1.3004749 -1.5208027 -1.5827719 -0.81070524 -0.78825974 -1.4955769 0.65427524 0.7773232 1.52854 -1.3704774 -1.5551959 -1.5273147 -0.8635994 -1.4388771 0.7067698 -0.8372018 -0.70699364 -1.2337446 0.64930755 -1.2505933 0.6223447 1.5348142 1.4140635 -0.68491936 -1.2581489 0.6726954 1.3267608 1.287502 0.87532127; 0.81092113 -1.3934917 -0.52509165 0.76498663 0.59978473 1.5781019 0.81333035 -1.3299788 -1.3683275 0.58423275 1.571548 1.5820303 -1.2928137 -0.6750458 1.5710756 -1.5174917 0.72958195 -0.5178763 -1.317027 -0.64780957 1.2376409 -1.3346989 -1.4369235 -1.2938073 -1.4945639 0.5797197 0.82199013 -1.2564309 -1.3375819 0.7905285 -0.74062055 -1.2428939 -1.3451716 1.3654428 0.84298444 -1.3567085 -1.5007137 -1.2678338 0.5820444 1.3044056 -0.8738267 -0.5703194 -1.4694525 -0.7769699 0.7266548 -0.74161553 -0.84070724 0.6623497 -1.2536069 -0.536299 0.7000681 0.60323 -0.7855618 -0.8173067 -1.47302 -1.5804603 -0.77034885 1.3113594 1.4682016 0.7349255 -0.6757929 -0.54717416 -0.79076606 -1.3179692 -0.7241815 1.3426976 -1.5732901 -1.4180417 -0.5253607 1.3007432 -0.82030934 1.4676316 0.51710784 0.7969778 -1.3939339 -0.62720776 1.5191842 0.78232795 0.8849005 1.3306173; -1.5210657 0.5560639 1.5909073 -1.3067204 -1.2672213 -0.64892316 -1.2744875 1.2697793 0.5872442 -1.2933912 -0.8018651 -0.6567132 0.8628649 1.4111214 -0.6497308 0.6167029 -1.2799039 1.2151182 0.7567851 1.4891493 -0.6204187 0.8261787 0.9034024 0.8681189 0.6068791 -1.2613417 -1.4049033 0.6552827 0.680625 -1.5818725 1.3531324 0.81033707 0.8280976 -0.8238955 -1.2597195 0.59742254 0.8674992 0.8307514 -1.3169601 -0.64577943 1.415194 1.3433673 0.92915124 1.4155977 -1.2517403 1.2154089 1.4244834 -1.5211129 0.8543465 1.5226887 -1.3572747 -1.4951824 1.3043362 1.2332091 0.78605205 0.8178592 1.3516985 -0.60261375 -0.86485714 -1.4869086 1.5420754 1.2337626 1.4168348 0.8991606 1.2950107 -0.73955643 0.6833973 0.8046783 1.2301419 -0.91790956 1.4143462 -0.7726409 -1.4681199 -1.2516582 0.72016317 1.3286338 -0.5565947 -1.4650805 -1.3220718 -0.75175565; 1.7571384 -0.39645696 -1.6024952 1.5111113 1.5107937 -0.23059812 1.8862658 -0.7202099 -0.16698201 1.4323616 -0.44382808 -0.10225477 -0.36895183 -1.706747 -0.14069651 -0.13158073 1.7474854 -1.38927 0.35720652 -1.6315262 -0.7449819 -0.41659087 0.23704092 -0.164179 -0.050384942 1.4991959 1.7813036 -0.7736764 0.69561505 1.6030886 -1.5201895 -0.3358921 -0.0456395 -0.7815751 1.4401971 -0.3548414 0.18628016 -0.31762707 1.7006869 -0.4449174 -1.3928515 -1.4248201 0.5733437 -1.8987569 1.5098954 -1.7447144 -1.522532 1.551312 -0.23657827 -1.4281925 1.7138848 1.6907865 -1.5130135 -1.6271684 -0.26660034 0.77486205 -1.3678037 0.13090935 0.63797814 1.685091 -1.7769114 -1.4822948 -1.6344528 -0.01680849 -1.693594 -0.069017604 0.43930617 -0.33781734 -1.4380705 -0.3289038 -1.5780251 -0.38693097 1.4755223 1.6086174 0.13545121 -1.6375024 -0.59525716 1.4717216 1.842082 -0.5003743; 1.5228834 -0.22864912 -1.6533171 1.4152253 1.7551624 -0.25027296 1.5191197 -0.6750923 -0.16616507 1.5914973 -0.14482544 -0.14539345 -0.43886718 -1.7082949 -0.22164728 -0.29173186 1.6415927 -1.4522091 -0.07409459 -1.4839928 -0.7222731 -0.25341338 0.24873263 -0.4179443 -0.24630798 1.4894365 1.7446675 -1.1015354 0.836191 1.4412752 -1.2932521 -0.5831445 -0.24013977 -0.39997846 1.6991606 -0.6733433 -0.11867585 -0.6316418 1.4851302 -0.34980714 -1.360005 -1.6704724 0.37215936 -1.3880144 1.6765838 -1.4259458 -0.9594988 1.47997 -0.46940437 -1.3657926 1.3642695 1.4610577 -1.6671208 -1.4485085 -0.48349872 0.6692827 -1.1142887 0.20771009 0.40486556 1.6703937 -1.5913937 -1.3971707 -1.7513876 -0.020803986 -1.1110439 -0.008872904 0.34640592 -0.15561095 -1.1693357 -0.04603714 -1.4278935 -0.11961314 1.3493345 1.3189489 -0.22248018 -1.2149522 -0.61406726 1.5901989 1.4300594 -0.2616527; -1.3507472 0.7916067 1.4837588 -1.3874265 -1.6143042 -0.5865086 -1.4539024 0.46950346 0.5117197 -1.5553172 -0.5637021 -0.6836123 0.5310234 1.6566858 -0.7723042 0.89802825 -1.5270997 1.4838688 0.6702227 1.6053355 -0.38839403 0.60088366 0.5872287 0.66341317 0.7676677 -1.5434214 -1.355515 1.2394372 0.37506437 -1.5525099 1.4654183 0.5397597 0.8075495 -0.4764987 -1.3925205 0.6766729 0.7898421 0.90133536 -1.3693391 -0.5836382 1.3205686 1.3704171 0.3979452 1.4597138 -1.5413015 1.2748563 1.2439702 -1.3093616 0.83655053 1.4976909 -1.5473297 -1.3548701 1.65635 1.6108271 0.5539802 0.43089345 1.1484667 -0.7311627 -0.8740072 -1.3577551 1.3703872 1.5314265 1.4258225 0.7165737 1.3484235 -0.5926808 0.7149665 0.7121928 1.460152 -0.46388376 1.4080085 -0.46770254 -1.501523 -1.5398346 0.71664405 1.2793328 -0.44829246 -1.6868802 -1.483963 -0.43597206; -1.5287162 0.11237473 1.1726438 -1.3275163 -1.4940698 0.8254205 -1.6647732 0.9720994 -0.051610723 -1.4301324 0.7858625 0.4979181 0.18141055 1.2966485 0.7801746 -0.05344917 -1.3050044 1.3419064 -0.7866848 1.3767473 0.9724649 0.14194195 -0.8345874 -0.27531642 -0.4635945 -1.1860583 -1.4579228 0.31048524 -1.0416428 -1.7316701 1.5629652 0.08794339 -0.4279992 1.1223576 -1.3173923 0.06426405 -0.6993544 -0.028143859 -1.6510065 0.9342187 1.570754 1.3600973 -0.9131117 1.7784249 -1.4206558 1.5171498 1.5785828 -1.636192 -0.16612042 1.4557437 -1.6402272 -1.5593649 1.5726922 1.3831543 -0.072168835 -0.9130275 1.6416167 0.2754772 0.06794588 -1.4288815 1.4093758 1.2930825 1.4320562 -0.48056003 1.6258662 0.6522743 -0.904808 -0.3404403 1.4223728 0.8065016 1.2343607 0.6818979 -1.7146734 -1.7646174 -0.5841223 1.5194643 0.7937757 -1.2013558 -1.5140848 0.9073683; 1.5747625 0.15064387 -1.4547994 1.4770261 1.2668813 -0.37894782 -0.45555663 0.561953 -0.5965607 1.2259107 -0.80252457 -0.7716849 0.032378037 -1.2678841 -0.53479415 -1.6069329 1.5893413 -1.3200389 0.9490542 -1.2050744 -1.0603131 0.78686064 0.23986493 -1.7128105 0.49470115 1.3347574 1.2878963 -1.716752 1.2220589 1.5214249 -1.5812975 -1.9567008 0.51329863 -1.3097082 1.3055246 -1.9337803 -0.2049533 -1.701096 1.5183854 -0.9207112 -1.1398265 -1.2843668 0.927143 -1.302548 1.4622295 -1.6135663 0.68363225 1.1043601 -1.9483347 -1.5255389 -1.1855862 1.2440445 -1.3538582 -1.4171848 0.67890215 1.2038697 -1.190906 -1.1219352 1.9038132 1.2344127 -1.14651 -1.5079045 -1.5957676 -0.30260187 -0.7389627 0.52792186 0.45575252 -0.19718719 -1.2740685 -0.8418817 -1.4517897 -0.98612505 1.6196094 0.2609773 -0.54504514 -1.3190219 -0.968944 1.269151 1.1001285 -0.99395066; 1.5617181 -0.76738036 -1.3102269 1.2678057 1.4856796 0.63822246 1.3795002 -1.2986923 -0.54212046 1.3082166 0.54067105 0.84413224 -0.6062743 -1.4960222 0.79866004 -0.67810994 1.4533083 -1.3985922 -0.6501611 -1.4244868 0.715578 -0.8447651 -0.5398597 -0.84228796 -0.7575798 1.3789634 1.4559464 -0.5253715 -0.6882594 1.4484236 -1.3529794 -0.78956753 -0.76429236 0.7806363 1.3731055 -0.77198917 -0.5644123 -0.5728147 1.5623164 0.7979282 -1.4701 -1.4739244 -0.64051026 -1.2666878 1.2548724 -1.4488556 -1.2649429 1.3560616 -0.6171905 -1.2720499 1.3261256 1.2881894 -1.4471319 -1.4885086 -0.77023846 -0.6564678 -1.1308476 0.63960284 0.70973516 1.3324739 -1.5218189 -1.361328 -1.336954 -0.74375576 -1.191958 0.5899662 -0.75592834 -0.696201 -1.1328444 0.76387984 -1.3555026 0.8774666 1.4206417 1.305182 -0.709121 -1.2891936 0.7660135 1.5367644 1.4722471 0.56272763; -0.9105062 0.7263278 0.237693 -0.37470856 -0.3971062 -1.8933698 -1.2887647 2.0767725 1.8012272 -0.20955764 -1.9220519 -1.0556239 1.798013 0.72302157 -1.5956019 1.1517621 -0.5223648 0.22819483 1.2568928 0.6309397 -1.5384753 1.8588097 1.2618172 1.0816236 1.0581387 -0.38146588 -0.5487937 0.74738586 1.5747275 -0.5020552 0.9791384 1.1651528 2.1079032 0.33588284 -0.42196152 1.1589277 1.886071 1.0615276 -0.72075 -2.1788995 0.22123057 0.25286457 1.7431809 1.0753268 -0.5305352 0.6827567 1.5687399 -1.139459 1.1048715 0.34663838 -1.7259538 -0.6586603 0.78143567 0.98327416 1.9289938 1.588798 1.2641028 -2.0282054 -0.6931518 -0.5662477 0.103198566 0.326861 0.33987543 1.5900673 1.3044261 -1.9729043 2.1867912 1.9476459 0.989648 -1.854115 0.41885677 -1.9262955 -0.44758144 -1.1532365 1.8451524 0.99242055 -1.9229568 -0.35728872 -1.2893449 -1.7886596; -0.6469306 1.4637809 0.5986307 -0.7149526 -0.6131352 -1.4217224 -0.78101534 0.97255224 1.3621033 -0.53906196 -1.2465503 -1.4618058 1.1341829 0.8697131 -1.5120057 1.4002062 -0.815033 0.55007833 1.5559074 0.8218387 -1.2735792 1.2994871 1.527391 1.2449985 1.2859076 -0.6959855 -0.7887105 0.9325571 1.3127286 -0.8703891 0.8100972 1.3562388 1.4515282 -1.6064527 -0.68804735 1.2104478 1.4309816 1.3433399 -0.5295818 -1.5193076 0.74396074 0.7240587 1.4156939 0.6966404 -0.74625266 0.6586287 0.84804577 -0.7739359 1.180515 0.6601185 -0.8291212 -0.6478637 0.7007669 0.7786726 1.2890679 1.366988 0.5908133 -1.1827434 -1.4479543 -0.5521919 0.54319096 0.8023517 0.71746 1.4798152 0.5966679 -1.4881076 1.5767725 1.4913977 0.6529617 -1.4713395 0.5758322 -1.572957 -0.6129255 -0.631078 1.5628403 0.7806425 -1.5785583 -0.6162794 -0.7283327 -1.5299864; 0.04084823 -1.177245 -1.1100513 1.605977 1.3176208 -0.49728873 2.3040335 -1.4618535 -0.055725407 1.5037534 -0.5161705 0.82741255 -0.6230302 -1.8396014 -0.07997851 0.39866215 1.273966 -1.2980098 -1.0900986 -1.9437939 -0.5230919 -1.1914082 0.16894253 0.10803645 -0.90177506 1.5473126 0.59116656 0.7947875 0.36401805 1.0299516 -0.55641073 -0.20920496 -0.5584087 -0.8615742 1.5182782 -0.18048523 0.60962933 -0.14480655 1.93143 0.21273622 -2.391712 -1.4267647 -0.4273794 -1.5084815 1.564001 -1.4212615 -1.6910888 1.8687537 0.17241763 -0.23153652 1.8795711 1.7617595 -1.6850692 -1.8309122 -1.3504953 0.113888964 -1.3285682 1.6281439 -0.027905416 1.5045513 -2.0478642 -1.4450265 -1.175139 0.19624388 -1.268124 -0.79226613 0.052318618 0.09618185 -1.0456102 0.44139576 -1.9023917 0.09521697 0.12762903 2.0502546 0.77421004 -1.4856355 -0.43656513 1.4996263 1.636159 0.31827328; 0.556683 -1.4031354 -0.48994532 0.32729608 0.31290102 1.5683256 0.70498407 -1.2731315 -1.283354 0.24349621 1.5070324 1.3561872 -1.2939584 -0.3707875 1.3835304 -1.2336873 0.5574579 -0.26866966 -1.3657146 -0.47020376 1.3830199 -1.3492002 -1.3416926 -1.2289523 -1.546045 0.1324089 0.76984805 -0.54238546 -1.4764874 0.39603853 -0.58764625 -1.3649282 -1.4726449 -0.17528522 0.30304515 -1.3567123 -1.5030353 -1.3456573 0.56234986 1.5327649 -0.17014064 -0.56585735 -1.4649415 -0.47896415 0.49761266 -0.6529946 -0.7050521 0.47137532 -1.4030542 -0.55383766 0.6886143 0.6055916 -0.38205367 -0.40185484 -1.5404179 -1.5489287 -0.75272197 1.2248344 1.1019151 0.6619739 -0.71054614 -0.48086366 -0.5107579 -1.5532948 -0.55975705 1.3438623 -1.2608075 -1.3932056 -0.58347756 1.6435497 -0.22613896 1.461572 0.7022679 0.5183804 -1.2974174 -0.74561983 1.694086 0.27248353 0.6877193 1.6863998; 0.7719163 -1.3964664 -0.31860164 0.6204947 0.5334933 1.2956597 0.5239249 -1.0608538 -1.4695861 0.37720013 1.5638769 1.4996326 -1.432937 -0.54064053 1.6229665 -1.3941653 0.5038745 -0.4904361 -1.508007 -0.44962963 1.561224 -1.5045764 -1.4469047 -1.2737207 -1.4647647 0.49223992 0.72876346 -0.6029361 -1.3931948 0.63942254 -0.67116344 -1.288782 -1.5335081 0.2002518 0.35169163 -1.3976849 -1.5769393 -1.197299 0.54800296 1.2671417 -0.475765 -0.5746916 -1.5684867 -0.8085773 0.2942573 -0.7356242 -0.7087289 0.44919875 -1.4747478 -0.7191927 0.6563911 0.42941797 -0.5106562 -0.6028354 -1.2377115 -1.5265489 -0.72796947 1.356143 1.2019451 0.45327622 -0.6462759 -0.3984558 -0.50754374 -1.5303427 -0.5878084 1.2996398 -1.4304299 -1.3564606 -0.5459447 1.5132476 -0.43452606 1.4049294 0.7500877 0.5664597 -1.221585 -0.45540264 1.3627732 0.40282327 0.48531944 1.5394689; -0.55412483 1.2822576 0.7987981 -0.86001974 -0.71770585 -1.4837097 -0.8066288 1.3757974 1.2185637 -0.49723426 -1.5484192 -1.5707724 1.2106116 0.6899171 -1.5783768 1.366868 -0.51286775 0.7392807 1.3306891 0.6362807 -1.4549894 1.3454846 1.586619 1.513046 1.4399318 -0.5735056 -0.5932939 0.84600306 1.6120852 -0.748925 0.8718403 1.4702415 1.5237693 -1.5947857 -0.72458667 1.3926518 1.3315738 1.4681244 -0.6348567 -1.3179096 0.71259093 0.81748384 1.2645409 0.8053424 -0.7215204 0.8290575 0.54676604 -0.53707844 1.5021119 0.6361419 -0.58848107 -0.5823427 0.80189806 0.5204567 1.2646344 1.4094231 0.74374235 -1.2240964 -1.2482418 -0.7625929 0.7478942 0.51534736 0.6582452 1.2696356 0.6365599 -1.4406244 1.5963427 1.2364455 0.767033 -1.5784928 0.8269824 -1.4253807 -0.77750796 -0.5426462 1.5004826 0.5026848 -1.364867 -0.6893846 -0.5384705 -1.5868075; -0.5641937 1.3567078 0.122355305 -0.4041579 -0.477582 -1.2992018 -0.45163542 0.74593955 1.3509318 -0.3788081 -1.528598 -1.4694088 1.0039697 0.27267042 -1.366418 1.1286291 -0.3114611 0.3245006 1.3411454 0.5224475 -1.3097279 1.2885702 1.517706 1.316097 1.4284362 -0.15857412 -0.39263213 0.91084635 1.5782255 -0.6483046 0.60084134 1.0715996 1.3506184 -0.17791076 -0.3354592 1.2759913 1.195644 1.2837788 -0.45000187 -1.3168591 0.19127041 0.2568485 1.5547066 0.6779875 -0.22761142 0.3370556 0.7684736 -0.537581 1.3701638 0.5920323 -0.5907205 -0.3936241 0.4300828 0.53230417 1.1500691 1.4298594 0.7020219 -1.2958412 -1.3575946 -0.22128308 0.52828526 0.32015055 0.6390416 1.3456125 0.6295941 -1.4455315 1.357322 1.3729894 0.60623974 -1.5006486 0.17639263 -1.4886135 -0.5890963 -0.6545028 1.4608998 0.5545016 -1.5837576 -0.2037949 -0.5799899 -1.4576087; 1.3659244 -0.5691136 -1.2903892 1.4534665 1.3686206 0.7194369 1.4168595 -0.76486063 -0.8241669 1.3987337 0.6864084 0.8501494 -0.7634366 -1.5734175 0.693422 -0.86648124 1.2773097 -1.3551179 -0.5621498 -1.3560055 0.53088564 -0.5219218 -0.7198133 -0.76682323 -0.71103394 1.5980172 1.5073041 -1.1211636 -0.7546675 1.2757077 -1.5166876 -0.7793398 -0.8485554 0.839331 1.5655656 -0.57796866 -0.6336954 -0.79633087 1.3063238 0.90151536 -1.3326131 -1.4894013 -0.73882496 -1.3357352 1.581888 -1.2267343 -1.2505677 1.2177025 -0.60687596 -1.4589752 1.3587867 1.4318904 -1.4503564 -1.2408171 -0.5974387 -0.71749634 -1.2525858 0.7490992 0.8651787 1.2867637 -1.4940492 -1.255383 -1.395474 -0.91145456 -1.3293488 0.55120975 -0.64728016 -0.8239968 -1.530162 0.5448214 -1.3132702 0.8582766 1.4824516 1.2193964 -0.5303876 -1.302174 0.60281473 1.5161085 1.3830022 0.83187735; -0.5550498 1.2888573 0.5611867 -0.57268435 -0.56429386 -1.5225838 -0.8368552 1.0140827 1.4042506 -0.4157982 -1.2498952 -1.3997152 1.3459778 0.5309488 -1.2761873 1.4592372 -0.56742567 0.40142393 1.5983607 0.5920574 -1.563322 1.3854796 1.6001799 1.1669221 1.361723 -0.4907796 -0.62874293 1.0997242 1.5181649 -0.6578991 0.69086874 1.206676 1.2533232 -0.51305294 -0.602212 1.1205541 1.3577677 1.4312981 -0.7174548 -1.2407696 0.6309408 0.7511413 1.598907 0.6532244 -0.6307472 0.6459318 0.79836494 -0.7651766 1.5417577 0.51866186 -0.47677898 -0.6159744 0.7394484 0.49529827 1.3465599 1.5939556 0.57628554 -1.2414931 -1.162101 -0.67287344 0.4696359 0.6294523 0.66963464 1.3668671 0.82889223 -1.573944 1.314532 1.2103453 0.6781961 -1.5988368 0.6777902 -1.4932802 -0.8307424 -0.5420391 1.4956392 0.6729759 -1.4270324 -0.42774114 -0.7294273 -1.616945; -1.4777919 0.3387565 1.6522013 -1.4287735 -1.5912195 -0.0067378078 -1.3813797 0.54894453 0.19620515 -1.4708774 0.2535226 0.14504291 0.2750519 1.4584053 -0.051063724 0.27695403 -1.551583 1.3683152 -0.009479463 1.6642971 0.5786004 0.22677526 -0.3127449 0.3915232 -0.0070526833 -1.693974 -1.5616723 1.0601612 -0.6015889 -1.7289516 1.4080544 0.5358076 0.23839316 0.6338484 -1.4905591 0.637777 0.055530906 0.7268435 -1.4699893 0.17289193 1.0608729 1.4312805 -0.41202804 1.4665744 -1.367377 1.5698171 0.9694171 -1.1863161 0.55869806 1.5053108 -1.2277455 -1.3319741 1.3529079 1.4383866 0.17909852 -0.59727305 0.86213815 -0.14544079 -0.52252537 -1.4329296 1.3137314 1.4040066 1.4853398 0.16849849 0.9154089 -0.16758527 -0.054459102 0.37826368 0.9746535 0.21355006 1.40418 0.13369095 -1.4837382 -1.2963896 0.2549307 1.124891 0.7427119 -1.6275735 -1.4887625 0.4566659; 1.3730643 -0.7561431 -1.3449519 1.6034726 1.3903078 0.5592249 1.6261561 -0.95444816 -0.5600562 1.3709707 0.5136313 0.55539 -0.85194755 -1.4361929 0.61049575 -0.8267566 1.6392821 -1.4820657 -0.59591544 -1.3423102 0.7006441 -0.61964685 -0.62011766 -0.90596217 -0.8687599 1.5057181 1.6297556 -1.6549675 -0.55343217 1.5153571 -1.2971834 -0.7505722 -0.87733334 0.6738811 1.6207747 -0.8962163 -0.55462193 -0.6904594 1.2911497 0.792154 -1.6771597 -1.4662081 -0.7643012 -1.3096552 1.3202704 -1.3178756 -1.376384 1.287437 -0.685101 -1.2726144 1.3157448 1.3483254 -1.2829291 -1.3274977 -0.8723229 -0.6560789 -1.5020102 0.51667434 0.62760353 1.3432744 -1.3323435 -1.6729172 -1.5375128 -0.84990275 -1.3631438 0.56028 -0.6313088 -0.7449856 -1.3668535 0.49123275 -1.3388544 0.6193013 1.4636652 1.5553889 -0.59066886 -1.2652632 0.3830046 1.360312 1.4375396 0.46189296; 0.8341644 -1.4159039 -0.412705 0.5534883 0.54215854 1.4662677 0.7969448 -1.2911801 -1.4648769 0.5387055 1.350564 1.4178522 -1.1690998 -0.6843042 1.4251598 -1.4502412 0.5908236 -0.6463921 -1.240453 -0.8293374 1.5972676 -1.5705867 -1.3999951 -1.4693633 -1.2350109 0.6586755 0.82019794 -0.69555795 -1.3120489 0.57059073 -0.5361339 -1.4715708 -1.2564853 -0.07628355 0.73035663 -1.212777 -1.5347395 -1.3722806 0.7115456 1.4376763 -0.58673024 -0.5688023 -1.5905979 -0.8071596 0.45798302 -0.54647833 -0.708899 0.78274345 -1.318488 -0.64682555 0.87800074 0.83682096 -0.7314751 -0.74401546 -1.4264058 -1.6431851 -0.5153126 1.4784274 1.3200895 0.47720483 -0.6573752 -0.6497241 -0.5932068 -1.550999 -0.79056937 1.4985492 -1.2636545 -1.3195529 -0.6639406 1.3273412 -0.66786885 1.5209304 0.51823425 0.642674 -1.3268 -0.61525875 1.3742146 0.6344151 0.5191418 1.3821126; 0.8857735 1.0500543 -0.92993253 1.0528136 1.2450397 -1.3393487 0.82465523 1.3330067 1.6410055 1.0651883 -1.4585205 -1.0267043 1.3721756 -1.044025 -1.3050956 1.3700306 0.8425658 -0.9598145 1.2777808 -1.15111 -1.3086483 1.2531462 1.2611892 1.6818212 1.1009527 0.90559375 0.8811836 0.11316116 1.279805 0.9419318 -0.5440948 1.1223449 1.253274 -0.9685547 1.2014935 1.5715681 1.4622899 1.483034 1.0298961 -1.3200438 -1.0205162 -1.0179441 1.3704823 -0.8403408 0.99425477 -0.7574094 0.32277054 0.65259314 1.4401611 -0.6632478 0.58629644 1.004347 -1.0418386 -0.99779004 1.5863248 1.1778252 0.016227433 -1.26965 -1.1469306 1.1634091 -1.1623867 -0.9939652 -1.2194752 1.5218462 -0.19287643 -1.3167456 1.3076761 1.5715638 -0.2232129 -1.4785733 -0.9319546 -1.4534422 0.93456846 1.0310185 1.1818938 -0.43171522 -1.3133354 1.0108567 0.85597515 -1.1861974; -1.5593768 0.9384508 1.4665602 -1.380293 -1.4910735 -0.80058277 -1.2279652 1.3116539 0.5609839 -1.3219633 -0.8381848 -0.7507197 0.78702533 1.4243723 -0.59845746 0.6765292 -1.23164 1.5900456 0.8809568 1.4043406 -0.64455104 0.7944894 0.79526925 0.5777969 0.7529405 -1.2646958 -1.4462289 0.7904412 0.8571367 -1.437051 1.289032 0.82532823 0.9330174 -0.64407784 -1.2790207 0.80759376 0.55832785 0.7670611 -1.2772899 -0.67686343 1.3151991 1.2547613 0.791961 1.4636298 -1.3922999 1.3158227 1.4715601 -1.4338315 0.6181922 1.4185172 -1.4442635 -1.5668997 1.3598168 1.2569959 0.834024 0.6803386 1.4350109 -0.9183823 -0.766413 -1.2226666 1.3059545 1.4841746 1.2749712 0.6619899 1.36561 -0.5716542 0.6552087 0.74363947 1.4535375 -0.734916 1.2457026 -0.5831629 -1.2541436 -1.3795182 0.5568615 1.5981055 -0.6716269 -1.3308578 -1.2270395 -0.79363245; -1.2279203 0.7077876 1.454631 -1.4597158 -1.3812336 -0.56710696 -1.288747 0.5489002 0.75912356 -1.42286 -0.8488141 -0.5616535 0.9135309 1.2385576 -0.59078395 0.557069 -1.6093215 1.2697521 0.53531396 1.3751358 -0.7636641 0.6522764 0.742953 0.8395755 0.6165059 -1.3077503 -1.3752422 1.3034799 0.5827458 -1.3644596 1.250585 0.86563355 0.63096076 -0.5850607 -1.3277949 0.57822603 0.5868509 0.86261153 -1.4742669 -0.8455292 1.5976629 1.5830584 0.55807203 1.4932902 -1.2848296 1.4915305 1.4333093 -1.2406466 0.54158723 1.5447154 -1.3805952 -1.2499354 1.59021 1.3350356 0.57952505 0.8446863 1.5023568 -0.8008413 -0.7717658 -1.4802213 1.5857227 1.4036318 1.5547773 0.9204218 1.5590255 -0.77921 0.56615525 0.56498617 1.331109 -0.74218607 1.3830262 -0.73843986 -1.4321177 -1.3506789 0.7713067 1.4903774 -0.69959104 -1.3700773 -1.4069713 -0.8109711; 0.58870876 -1.0959675 -0.70795006 0.5034228 0.53758204 1.5769243 0.8301181 -0.67164683 -1.2532456 0.457637 1.4339226 1.4272714 -1.3706882 -0.70024866 1.5108521 -1.2639358 0.7158842 -0.52740407 -1.5589309 -0.5438391 1.2932991 -1.3018992 -1.5859004 -1.4764518 -1.3443888 0.74152213 0.663878 -1.2457865 -1.2444869 0.8177301 -0.5836282 -1.4044925 -1.2417809 -0.4597405 0.67428905 -1.3380249 -1.4705046 -1.2861507 0.6604172 1.2770929 -0.44091526 -0.61576426 -1.4213734 -0.68601996 0.48963618 -0.47751945 -0.58388656 0.69036114 -1.408721 -0.8331164 0.76041317 0.5693785 -0.80025387 -0.7930516 -1.1836979 -1.5163922 -0.6070055 1.4211779 1.1966612 0.7627182 -0.6675708 -0.6461835 -0.76651734 -1.403811 -0.54621524 1.4302475 -1.3340106 -1.2224094 -0.5677498 1.5183179 -0.46492592 1.5350605 0.49094856 0.68912035 -1.5396192 -0.78699136 1.6049527 0.7231976 0.8093401 1.5809761; -0.6456501 -1.3342141 0.95601237 -0.9392508 -0.91064936 1.1602082 -0.9191949 -1.548611 -1.7289131 -1.2351226 1.4630736 1.353086 -1.5362657 1.278177 1.3723559 -1.5720718 -1.0193409 1.1411109 -1.2987446 0.99254346 1.4033636 -1.2786316 -1.4035685 -1.7772081 -1.2741562 -1.0712367 -0.823812 -0.17608774 -1.2932352 -0.9563415 0.68443775 -1.4264504 -1.2723521 0.9059948 -1.1611441 -1.6517259 -1.3451678 -1.3916159 -0.9564076 1.3239838 1.0197248 1.2425243 -1.2642169 0.906891 -1.2773396 0.95653075 -0.90859157 -0.77630424 -1.388218 0.9208882 -0.5919394 -1.1382651 0.8865336 0.96464694 -1.6247146 -1.3705438 -0.21938775 1.3405467 1.505526 -0.90101296 1.0053567 1.0037944 0.9023286 -1.5730144 0.20508316 1.368948 -1.2766851 -1.4288712 0.27150333 1.305662 1.0561644 1.3029087 -0.7825084 -0.9956617 -1.3907205 0.4816096 1.373851 -0.994852 -0.72525126 1.4035296; -0.8156227 0.41137102 1.4015467 -1.4892097 -1.4778311 0.8441098 -1.809038 1.2221175 -0.25847593 -1.3447717 0.80523807 0.41506165 0.40661195 1.5783609 0.6066892 -0.37974104 -1.628904 1.4250524 -0.13079256 1.6775997 0.98989564 0.72716147 -0.9135151 -0.34876397 -0.0011408052 -1.5603695 -1.393982 -0.11115269 -1.0127186 -1.3711541 1.1569085 -0.44641322 -0.17643209 1.1492318 -1.4831854 -0.33909407 -0.40462038 -0.35473907 -1.5336254 0.7056681 1.3926468 1.540825 -0.47027695 1.6604574 -1.6454315 1.3206338 1.4509609 -1.5994552 -0.4752124 0.53728294 -1.9388493 -1.7797635 1.4202043 1.5623934 0.4419809 -0.8287354 1.1338558 -0.56111866 0.10478857 -1.6214918 1.5866678 1.4843733 1.5647848 -0.5055371 1.561939 0.670501 -0.84429985 -0.13031302 1.1880324 0.2671654 1.4915421 0.54344445 -1.0955293 -1.6949036 -0.64277714 1.5183489 1.109544 -1.4820119 -1.4715961 0.62261957; 0.40145928 -0.44628483 -0.42113155 -0.11033477 -0.121971175 -0.08765537 -2.7703612 2.8586357 0.019045368 0.031505834 -0.4206548 -0.7661576 0.9933269 1.1634387 -0.08213072 -0.9380104 0.21440005 -0.4773047 0.75141454 1.2969427 -1.6313833 1.4003433 -0.42777616 -2.6252692 -0.082225904 0.38647473 0.9344829 -0.4025177 1.9375882 0.90211326 -0.4440825 -1.7641764 1.6657423 -0.03653472 -0.029686749 -1.5067029 -0.85368013 -1.8505664 -1.000221 -2.082291 0.48601943 0.3861147 1.7294891 1.3741661 -0.5558906 0.014946337 3.9993584 -2.5991552 -3.3251388 -0.9762091 -4.8908587 -1.3282036 -0.24611974 1.5564584 1.2986566 1.74277 2.5090775 -2.5609324 2.4727736 -0.9571733 0.7352998 -0.43738723 -1.076514 -0.67669994 2.5552595 0.7195499 0.6111728 0.0885396 0.89974546 -1.0289435 0.25992757 -2.1406648 1.2609576 -3.309497 -0.6206991 1.5868855 -1.7002914 0.14277987 -2.244404 -1.1999302; 0.33979607 -1.3257002 0.19721812 -0.28614247 -0.2800723 1.3594565 0.4712604 -1.1515057 -1.1800004 -0.5086487 1.3549728 1.5592918 -1.2364091 0.023073243 1.4214177 -1.0142589 0.060434893 0.08576407 -1.5625277 -0.027148211 1.6776274 -1.2596774 -1.5924456 -1.3196999 -1.5460232 -0.24168164 0.3138469 -0.8247297 -1.7902683 0.3371517 -0.2826391 -1.0490195 -1.1668749 0.6805699 0.025301177 -1.2415614 -1.1034449 -1.1609795 -0.045062445 1.5621095 0.42088544 0.047800988 -1.5363588 -0.4807441 -0.060150556 -0.30175292 -0.56475216 0.4715683 -0.9772135 -0.36482257 0.3071369 0.038691495 -0.321197 -0.12944563 -1.2463305 -1.7584281 -0.7112823 1.4617202 1.1363345 -0.013940863 -0.17240633 0.10414214 -0.14875725 -1.3037603 -0.4454637 1.3626039 -1.4945925 -1.1213801 -0.36049503 1.3502241 0.22290054 1.4754307 0.41301614 0.34904286 -1.1347473 -0.30566344 1.6419014 -0.21909569 0.20269479 1.4702832; -1.5731802 0.72942877 1.2782452 -1.4885383 -1.416864 -0.7396806 -1.2313775 0.7313599 0.6777482 -1.3228891 -0.8997277 -0.87477463 0.6997974 1.4211394 -0.86905676 0.7997158 -1.528608 1.4797263 0.7151707 1.3503383 -0.5996645 0.8940771 0.9069226 0.6580025 0.6636542 -1.3974113 -1.5734799 1.4550899 0.7078566 -1.5667251 1.3523463 0.8637766 0.5816977 -0.7948961 -1.3043747 0.5926511 0.9188443 0.6273959 -1.3860202 -0.7946333 1.3070841 1.5611286 0.6183378 1.3514128 -1.2354455 1.2654662 1.3411837 -1.2649845 0.8355278 1.3244425 -1.2943112 -1.4194243 1.391388 1.5343741 0.7582955 0.5156304 1.2231214 -0.65080523 -0.7596135 -1.3481791 1.3736361 1.5613322 1.4522482 0.6508564 1.4375857 -0.8625757 0.5843858 0.69225854 1.5854335 -0.67667615 1.2473913 -0.5892508 -1.3248926 -1.5899186 0.5869518 1.2166938 -0.66202843 -1.242561 -1.2974647 -0.6881172; -0.6742974 1.3508878 0.55704767 -0.70393485 -0.61584985 -1.5888371 -0.62408316 1.126682 1.4209301 -0.5458077 -1.3579739 -1.6104425 1.5299553 0.72797775 -1.5288008 1.3377776 -0.80680484 0.49150944 1.5368762 0.5554112 -1.5025564 1.2450293 1.2801317 1.3695211 1.3399875 -0.80201346 -0.70969164 0.7840216 1.3293434 -0.7007744 0.83921987 1.3540645 1.2964185 0.10578786 -0.7464914 1.1011037 1.5207058 1.3526332 -0.4978969 -1.2866912 0.5108687 0.47641474 1.2621522 0.8056558 -0.593362 0.73838484 0.8651859 -0.5198872 1.3736161 0.7256906 -0.86372054 -0.87079734 0.8247419 0.5145665 1.5521748 1.5677115 0.7877324 -1.4582317 -1.3764731 -0.5212834 0.8039651 0.48736876 0.6500838 1.2045033 0.7666814 -1.2241846 1.5809859 1.2791171 0.7877316 -1.3483336 0.7015247 -1.2684842 -0.7021016 -0.6400515 1.3238333 0.73935187 -1.6068699 -0.5775857 -0.8156157 -1.4085388; -1.3504965 0.65863186 1.290524 -1.4923353 -1.3828673 -0.89804995 -1.421813 1.5223961 0.74435306 -1.2753164 -0.66387105 -0.8971039 0.6552472 1.2441446 -0.59095657 0.5736286 -1.4114699 1.3325789 0.8995805 1.2534181 -0.7100313 0.7279509 0.8925411 0.8780106 0.7915784 -1.3160748 -1.4101869 0.83355284 0.6907211 -1.4573538 1.4553213 0.8706177 0.66626424 -0.9322724 -1.4213407 0.67808175 0.8714024 0.6885013 -1.4717367 -0.9151148 1.593215 1.2790521 0.88605404 1.2080629 -1.4982626 1.2684903 1.4232167 -1.5197893 0.7669972 1.2401497 -1.2765594 -1.4987103 1.207608 1.3677573 0.8422897 0.7785583 1.4739299 -0.56868744 -0.73038816 -1.3180245 1.385012 1.4445219 1.3867259 0.84757936 1.4691072 -0.64673567 0.6126346 0.5788912 1.4503192 -0.8365248 1.5023952 -0.5900694 -1.3159336 -1.3748031 0.5986676 1.37442 -0.6935943 -1.3329704 -1.2616534 -0.70348126; 1.2323626 -0.4922287 -1.5834053 1.5475719 1.4285638 0.61275035 1.351443 -1.0064481 -0.728173 1.6385803 0.7282188 0.6624172 -0.5078969 -1.5037962 0.5236344 -0.5919021 1.593047 -1.7063433 -0.6781989 -1.4646826 0.56890756 -0.48696196 -0.33432305 -0.82385623 -0.5715174 1.6111571 1.4535931 -0.92897725 -0.34258237 1.5002956 -1.4179316 -0.6400323 -0.4213339 0.44660383 1.5569803 -0.40432802 -0.5102491 -0.5205431 1.6162632 0.7604 -1.0848697 -1.5513616 -0.30793688 -1.2679003 1.5516446 -1.5851218 -1.3035463 1.3250928 -0.5233487 -1.1582417 1.3704494 1.3423187 -1.2801894 -1.4050448 -0.65102226 -0.550588 -1.3077027 0.4567403 0.6542952 1.5984582 -1.4031723 -1.6297884 -1.4943204 -0.7809311 -1.2383785 0.7334019 -0.7336235 -0.519016 -1.0857657 0.5427612 -1.2856933 0.5612792 1.2591302 1.4725845 -0.6671682 -1.2152236 0.42768317 1.5062834 1.2704717 0.6589751; -0.46211466 1.2669269 0.32058808 -0.5514877 -0.4028332 -1.2548414 -0.6387215 1.226864 1.3369538 -0.31705302 -1.3209591 -1.4410269 1.4457687 0.7418914 -1.4749187 1.1818705 -0.6318824 0.5821009 1.5578524 0.64391804 -1.5849277 1.4112772 1.3951521 1.4725969 1.5596039 -0.30210924 -0.82587624 0.8128521 1.5546542 -0.67477745 0.65083855 1.1307214 1.4997203 -0.2167348 -0.72944987 1.3259078 1.5371025 1.3735334 -0.58145064 -1.5208138 0.41583192 0.5418996 1.4695823 0.65241504 -0.6046616 0.79750186 0.6890868 -0.5530416 1.3712314 0.6132665 -0.6037263 -0.67363703 0.4378127 0.48361468 1.3714111 1.4105191 0.7689117 -1.5324188 -1.4653265 -0.54658806 0.7559694 0.47346357 0.49008074 1.2528944 0.5739789 -1.4931372 1.5009097 1.4748651 0.7844786 -1.2888254 0.44517213 -1.3876044 -0.794461 -0.6025848 1.2898251 0.7037709 -1.3516432 -0.5561258 -0.57157046 -1.4785234; -1.550763 0.88962704 1.4640033 -1.5093538 -1.5137291 -0.9055532 -1.0544931 0.6200426 1.0094196 -1.2660786 -0.8551693 -0.7649723 0.8226631 1.0543315 -0.87771887 0.7870214 -1.4825051 1.2504833 0.81994414 1.333064 -0.4563576 0.8635281 0.62395424 1.0850544 0.64653426 -1.4806268 -1.4187297 1.4539937 0.672178 -1.4064858 1.4143472 1.3097298 0.68040544 -0.7151001 -1.4213126 1.1189289 0.75124705 1.4814649 -1.0537307 -0.7956759 0.9991732 1.2707587 0.7657298 1.0273718 -1.3707583 1.33999 0.5831676 -1.1107117 1.028043 1.2648413 -0.94590586 -1.1628441 1.501205 1.3887364 0.8199089 0.65326595 1.1087424 -0.673507 -1.3061337 -1.1577821 1.0736202 1.6223857 1.3250347 0.68991387 0.8913797 -0.8807882 0.8615966 0.73824143 1.1673764 -0.67063266 1.3755442 -0.5780769 -1.4537504 -0.95119876 0.71637344 1.1148462 -0.41928774 -1.2277216 -1.1519076 -0.81875; 0.5569476 -1.4776351 -0.60549843 0.5821151 0.73505193 1.4213717 0.6896941 -1.4219583 -1.4806337 0.72276115 1.4423195 1.590342 -1.3486205 -0.74435437 1.3651392 -1.3999186 0.8425158 -0.73791504 -1.4284986 -0.84681255 1.3926741 -1.3588363 -1.345082 -1.3824911 -1.318833 0.8295258 0.6988597 -0.6723979 -1.5351621 0.67544997 -0.68704265 -1.2578974 -1.2430313 1.53826 0.5494933 -1.3330706 -1.4698542 -1.3946697 0.88186437 1.5866584 -0.7200384 -0.810458 -1.2873945 -0.5627907 0.84436226 -0.79091185 -0.56493413 0.89465857 -1.3141122 -0.80600834 0.68287015 0.77279985 -0.6933254 -0.7508191 -1.3680553 -1.3036498 -0.714251 1.4995215 1.4322964 0.7444539 -0.7163456 -0.5879105 -0.7536722 -1.2160141 -0.86615765 1.3677511 -1.3023134 -1.5213679 -0.5939321 1.3981895 -0.55181134 1.3694359 0.6258428 0.8994739 -1.4465455 -0.7787669 1.3780305 0.61529684 0.70200485 1.4909729; 1.2261727 -0.6999459 -1.4388863 1.340359 1.4115775 0.8495777 1.5654892 -1.0409737 -0.8305939 1.5288713 0.88000196 0.7665699 -0.839442 -1.3556787 0.8250931 -0.655591 1.439397 -1.4101743 -0.8301005 -1.3377936 0.820682 -0.5576599 -0.5033035 -0.87341654 -0.7894192 1.2847955 1.3753318 -0.73126554 -0.72398525 1.4432701 -1.3774232 -0.50595325 -0.52513885 0.5152 1.3549901 -0.7370586 -0.67572737 -0.7317324 1.3162042 0.6662776 -1.3972292 -1.5724157 -0.80646294 -1.4179857 1.5215782 -1.5390216 -1.2972549 1.3474313 -0.5378647 -1.2395114 1.5146241 1.2171243 -1.3079448 -1.5541325 -0.51733387 -0.5842724 -1.2043419 0.55364925 0.720607 1.2808194 -1.2896739 -1.3112372 -1.5527685 -0.69497687 -1.341495 0.5114665 -0.86719763 -0.6751173 -1.2847544 0.6198055 -1.5497792 0.7805176 1.150663 1.3508942 -0.5378859 -1.1498177 0.58744836 1.270015 1.4086385 0.8363402; 1.4394566 -0.43546402 -1.6632196 1.7916024 1.6194164 -0.021520942 1.5100707 -0.82641584 -0.6095376 1.6626861 -0.23652638 -0.03497337 -0.540307 -1.5191563 0.17934924 -0.3195257 1.5165275 -1.5441502 0.096809864 -1.766042 -0.58119404 -0.5587667 -0.0555044 -0.35810894 -0.2885482 1.4981229 1.5815012 -1.2840666 0.64148986 1.5289398 -1.434842 -0.73715943 -0.13490605 -0.4641842 1.5572605 -0.66846585 -0.12351964 -0.58066547 1.5800588 -0.23622724 -1.4572561 -1.7005737 0.34416956 -1.4797546 1.5364274 -1.5684606 -1.3385202 1.5380945 -0.6342365 -1.6695666 1.627237 1.5467387 -1.8244004 -1.6760297 -0.6723431 0.5365171 -1.3229841 0.41615897 0.51916933 1.5005147 -1.5669 -1.6140956 -1.6852121 -0.13684094 -1.3963299 0.2554753 -0.06168546 -0.27386585 -1.5374807 0.18604997 -1.5213921 -0.16589904 1.5058404 1.5186567 -0.055772223 -1.4978263 -0.5516514 1.541725 1.6582514 -0.24016279; 1.299416 -0.6481683 -1.5892801 1.3907248 1.4533355 0.61635715 1.4621072 -0.60483164 -0.8302102 1.4983205 0.5585401 0.71965617 -0.6960703 -1.5474147 0.58438206 -0.650124 1.3845196 -1.3814319 -0.8344908 -1.5852859 0.6636061 -0.6932888 -0.66992074 -0.6704244 -0.89735454 1.3127837 1.5035354 -0.5825448 -0.6070244 1.3356222 -1.5604453 -0.7031526 -0.8719941 0.86536354 1.3222854 -0.65889776 -0.6172801 -0.5674671 1.3546455 0.7517671 -1.4704502 -1.254881 -0.77523 -1.5502234 1.4619964 -1.5232924 -1.5013112 1.243556 -0.8339301 -1.5130651 1.5767341 1.2500676 -1.3644657 -1.278615 -0.6283312 -0.66769403 -1.3460097 0.7146973 0.61803454 1.4015363 -1.482083 -1.422682 -1.4992932 -0.54720694 -1.4319626 0.8786414 -0.6297995 -0.63653 -1.5689061 0.87020886 -1.2893981 0.6285191 1.2942262 1.3159844 -0.7003144 -1.3273966 0.882617 1.2694476 1.3456367 0.5654365; -0.7442837 1.2833395 0.5641544 -0.5683755 -0.79900545 -1.6086113 -0.5904696 1.3559698 1.4479373 -0.61833256 -1.6013111 -1.5848793 1.2337497 0.4660292 -1.2845777 1.1968219 -0.728169 0.5766226 1.4647814 0.53590035 -1.5122728 1.2779524 1.2654436 1.1942886 1.5353016 -0.57861966 -0.5556039 0.50717795 1.4891227 -0.5457259 0.7675221 1.1890931 1.3752657 -0.10031355 -0.60296446 1.300672 1.3458831 1.1645339 -0.8365753 -1.3969953 0.6933443 0.52622235 1.3553072 0.59361976 -0.79964155 0.79343534 0.5501773 -0.49842256 1.3249896 0.61496145 -0.53586125 -0.8231468 0.49799588 0.51492536 1.3899356 1.6205039 0.7963891 -1.4012222 -1.4150593 -0.6364817 0.78663 0.58966815 0.8009816 1.5167038 0.5603588 -1.2792228 1.3899786 1.5198771 0.70974404 -1.5166954 0.72444004 -1.2576288 -0.6458496 -0.8818098 1.5301683 0.64566284 -1.5951955 -0.5601681 -0.50097346 -1.5568448; -0.81946564 -1.5182823 1.3306332 -0.99188733 -0.90086645 1.4727912 0.40076476 -1.0781858 -0.9934316 -1.2305951 1.2458556 1.5900526 -1.1046448 0.56321853 1.4985758 -0.6245697 -0.9814196 1.2176988 -1.7057481 0.8049583 1.5984479 -1.4067141 -1.3318727 -0.5404574 -1.4496052 -1.1501596 -0.9538089 1.2147197 -1.3572006 -0.77979016 0.5084259 -0.19482249 -1.3075521 1.5506955 -1.1386575 -0.19831385 -0.87069184 -0.020671992 -0.71936804 1.3928267 0.72652346 0.9802105 -1.5458868 0.5378093 -0.77934164 0.57939374 -0.6757324 -0.15357654 -0.5278886 0.6209685 0.8022373 -0.62310743 0.6426002 0.58077335 -1.3423811 -1.3890688 -0.07038913 1.5802023 0.8402352 -0.6222649 0.66531384 1.2055975 1.1634371 -1.222421 -0.020916909 1.1104369 -1.4622014 -0.9537054 0.17634006 1.6700569 0.857171 1.5647948 -0.7877953 -0.06914773 -1.1176429 0.013412194 1.6575445 -1.1524346 -0.23620129 1.6760457; 0.6634206 -1.4653492 -0.5838347 0.5950553 0.60070664 1.4844238 0.67461354 -1.4415495 -1.311036 0.6259652 1.3739613 1.4247235 -1.3739353 -0.7803192 1.3160186 -1.2633073 0.769407 -0.86038285 -1.4517219 -0.7136459 1.390592 -1.3896241 -1.5324008 -1.3619895 -1.5586333 0.66065896 0.5795359 -0.57053566 -1.4537468 0.7526355 -0.87605846 -1.2579737 -1.3735191 1.6694418 0.55427814 -1.357771 -1.5177338 -1.3817936 0.8293223 1.3819053 -0.7059497 -0.79723334 -1.5337687 -0.7157305 0.8006519 -0.53833383 -0.61733353 0.5510314 -1.4599037 -0.7987996 0.84903336 0.7428083 -0.6356968 -0.7159804 -1.3325993 -1.4541472 -0.7015477 1.2094793 1.2133936 0.5473791 -0.72940964 -0.64031667 -0.7780496 -1.5346854 -0.5219554 1.3214966 -1.5567725 -1.4902997 -0.60981643 1.4444517 -0.7009069 1.3840324 0.5592489 0.62765497 -1.566617 -0.51867133 1.4075501 0.7759612 0.5481061 1.591741; 0.98781824 -0.47540042 -0.36948326 0.53273016 0.41912642 1.8449491 1.057295 -2.0890543 -1.8998873 0.48230398 2.1962576 0.8692067 -1.8188587 -0.6796982 1.669352 -1.8462596 0.6276264 -0.36374325 -1.0352606 -0.6704841 1.4114754 -1.8279908 -0.98838323 -1.480995 -1.0414485 0.117773235 0.6017887 -0.72534806 -1.3325133 0.45618057 -1.0467433 -1.0233356 -1.8809153 -0.51100355 0.49036643 -1.537545 -1.9822072 -1.4073584 0.8326805 2.2429883 -0.18541832 -0.34828845 -1.939757 -0.8798801 0.5752528 -0.89626366 -1.7614458 1.2272913 -1.0714258 -0.36710122 1.9203248 0.7365183 -0.556616 -0.7522554 -2.3422418 -1.3670619 -1.3336225 2.169789 0.88228345 0.81076384 0.2597391 -0.40007532 -0.25240603 -2.009635 -1.3118149 2.0625823 -2.185814 -2.2095118 -1.1100303 1.9017061 -0.48591283 1.8961468 0.51279217 1.1972349 -1.9106323 -1.3303665 1.5691031 0.08060067 1.1865076 1.8649205; -0.23303041 -1.4059577 0.8828895 -1.0270829 -1.1577108 1.3060895 -0.5813457 -0.40777192 -1.418736 -1.0876791 1.5605841 1.5875932 -0.91814095 0.8353684 1.3555148 -1.356766 -0.9262269 0.9435035 -1.5949036 0.844091 1.4074998 -1.1057281 -1.4874921 -1.1343507 -1.4899977 -1.2571067 -0.5684593 -0.7660131 -1.4944503 -0.33808988 -0.11478202 -1.0530328 -1.5443585 1.3641416 -1.0599577 -0.955561 -1.5865493 -1.2340928 -0.95035994 1.4817772 1.2225105 1.0077547 -1.4638717 0.63127726 -0.9138027 0.45726934 -0.16389947 -0.52301675 -1.1076312 0.15702115 -0.41347644 -0.8213123 0.923054 0.95555055 -1.1848176 -1.4311011 -0.08759044 1.4595865 1.2645264 -0.7874432 0.96128 0.9721427 0.8478843 -1.5822743 -0.1147803 1.4592955 -1.4586102 -1.122733 0.024351228 1.3088102 1.2631817 1.3848454 -0.37034848 -0.48460928 -1.4353266 -0.016761826 1.5069733 -1.2027475 -0.33919096 1.5933896], bias = Float32[0.23658118; -1.5218446; 1.4341401; -2.245755; 0.72126865; 1.4860463; -0.23010291; -1.458022; -0.6574635; 1.0353518; -1.316893; -0.32655844; -0.57105255; 1.5144129; 0.45442197; 1.1923661; 1.1772898; -0.33908328; 1.4406943; -0.8162934; -0.49041927; -0.6141595; -0.7066298; 0.688311; -0.8235948; -1.0853014; -0.31958833; 0.09483737; -0.79633546; -0.8232175; 0.21865454; 0.67724186; 1.4677398; -0.73936474; 0.84437144; -0.14728981; 0.6900897; 1.4373164; -0.72403854; -1.0342867; -1.0352176; -0.22669706; 1.144972; -0.46035472; 0.61693585; -1.9252989; -1.4464282; -0.6307905; 1.5723834; 1.5341129; -1.4537897; -1.3128394; 0.6744252; -1.5106959; 1.0195878; 0.37344086; 1.5133508; -1.416305; -0.7146418; -0.6517453; 1.426815; 1.4100777; 0.6681253; -1.1481172; 1.1415715; -0.6629421; -1.4862754; -0.7299036; 0.25204098; -1.5426878; -0.85541224; 1.43469; 0.5862043; -0.9156639; 0.6532396; -1.5133358; 0.86892164; 1.4711878; 2.109423; 1.1110657]), layer_3 = (weight = Float32[0.7251024 0.91203547 0.88387084 0.70494187 0.9516374 -0.7552263 0.6987498 0.63718015 -0.8721128 0.59715366 0.68731034 0.8057158 1.1164474 0.6371571 0.9356379 -0.7208995 -0.699547 1.1177337 0.81812066 0.5702566 1.1188142 -0.9663393 -0.6189103 0.9278575 -0.48456445 -0.73625267 1.119369 -0.83862036 -0.82638186 -0.7870831 0.88401425 0.3757475 0.6404573 -0.43210587 -0.5431552 -0.72214854 -1.0846282 0.5092094 -0.98168725 0.9677739 0.7408111 -0.80800533 -0.95289034 -0.8177931 0.8923439 0.7524662 -0.55315626 -0.5220752 0.70129377 0.5601714 -0.61081886 -0.63749045 0.85372436 -0.5062782 -0.6954403 0.9886468 0.6774791 0.647032 -0.9922229 -0.90515786 0.5256166 -0.61918527 0.4794284 0.6153802 0.7332485 -0.93375945 -0.7568629 -1.1273397 0.815988 -0.5434964 -0.67000425 0.833976 0.8821756 0.84023154 0.92068607 -0.68331003 0.7258939 0.73641473 -0.5212299 0.5981072; 0.9179792 1.438604 0.5393373 0.719365 0.5364796 -0.7717111 0.9042573 0.93351483 -0.74849224 0.7585406 0.9574124 0.86737734 1.4580708 0.6003847 0.84400064 -0.86382014 -0.93188083 1.4745628 0.52205586 1.0088887 1.5450742 -0.9205121 -0.9617998 0.78539973 -0.82868457 -0.6645012 1.4823223 -0.72010237 -0.59318703 -0.6261218 0.62756693 1.0431999 0.71335983 -0.8608723 -0.9540243 -0.85860807 -1.5841084 0.55514246 -0.5399748 0.94666684 0.86027306 -0.9326216 -0.91266865 -0.60818493 0.66271657 0.74828625 -0.6161438 -0.853154 0.61311406 0.65495336 -0.5587366 -0.80758464 0.85189676 -0.61759377 -0.86518025 0.94579047 0.5235632 0.92089295 -0.60142004 -0.8906354 0.7815364 -0.96115726 0.84222054 0.87368923 0.7499763 -0.8822814 -0.53283876 -0.2893159 0.7037517 -0.5896641 -0.8563918 0.5388784 0.66137064 0.9593479 0.8025387 -0.5624633 0.6619944 0.49844494 -0.8768113 0.6540259], bias = Float32[-1.4300042; -1.4166468]))

loss_neuralode(p_trained_high)